package yamake

import (
	"fmt"
	"io"
	"os"
)

const (
	MacroVersion             = "VERSION"
	MacroOwner               = "OWNER"
	MacroProgram             = "PROGRAM"
	MacroLibrary             = "LIBRARY"
	MacroUDF                 = "UDF"
	MacroYqlUDF              = "YQL_UDF"
	MacroDLL                 = "DLL"
	MacroGoProgram           = "GO_PROGRAM"
	MacroGoDLL               = "GO_DLL"
	MacroGoLibrary           = "GO_LIBRARY"
	MacroGoTest              = "GO_TEST"
	MacroPyProgram           = "PY_PROGRAM"
	MacroPy2Program          = "PY2_PROGRAM"
	MacroPy3Program          = "PY3_PROGRAM"
	MacroPy23Program         = "PY23_PROGRAM"
	MacroPyModule            = "PYMODULE"
	MacroPy2Module           = "PY2MODULE"
	MacroPyLibrary           = "PY_LIBRARY"
	MacroPy2Library          = "PY2_LIBRARY"
	MacroPy3Library          = "PY3_LIBRARY"
	MacroPy23Library         = "PY23_LIBRARY"
	MacroPython3UDF          = "YQL_PYTHON3_UDF"
	MacroPythonUDF           = "YQL_PYTHON_UDF"
	MacroJavaProgram         = "JAVA_PROGRAM"
	MacroJavaLibrary         = "JAVA_LIBRARY"
	MacroJavaContrib         = "JAVA_CONTRIB"
	MacroJavaContribProxy    = "JAVA_CONTRIB_PROXY"
	MacroExternalJavaLibrary = "EXTERNAL_JAVA_LIBRARY"
)

const (
	TargetUnknown YMakeTarget = iota
	TargetGenericProgram
	TargetGenericLibrary
	TargetGenericDLL
	TargetGenericUDF
	TargetGoProgram
	TargetGoLibrary
	TargetGoDLL
	TargetGoTest
	TargetPyProgram
	TargetPyLibrary
	TargetPythonUDF
	TargetJavaProgram
	TargetJavaLibrary
)

type (
	YMakeTarget int

	YMake struct {
		TargetType YMakeTarget
		Owners     []string
		Name       string
		Version    string
	}
)

func ParseFile(filePath string) (*YMake, error) {
	f, err := os.Open(filePath)
	if err != nil {
		return nil, err
	}
	defer func() { _ = f.Close() }()
	return Parse(f)
}

func Parse(r io.Reader) (*YMake, error) {
	ya := &YMake{}

	tok := newTokenizer(r)
	haveDeclaration := false
	haveVersion := false
	haveOwner := false

	parseDeclaration := func(m *yMacro, targetType YMakeTarget) error {
		if haveDeclaration {
			return fmt.Errorf("duplicate declaration macro: %s", m.Name)
		}

		if len(m.Args) > 0 {
			ya.Name = m.Args[0]
		}

		ya.TargetType = targetType
		haveDeclaration = true
		return nil
	}

	parseVersion := func(m *yMacro) error {
		if haveVersion {
			return fmt.Errorf("duplicate version macro: %s", m.Name)
		}

		if len(m.Args) > 1 {
			return fmt.Errorf("multiple versions in %s() macro: %v", m.Name, m.Args)
		}

		if len(m.Args) < 1 {
			return fmt.Errorf("empty %s() macro: %v", m.Name, m.Args)
		}

		ya.Version = m.Args[0]
		haveVersion = true
		return nil
	}

	parseOwner := func(m *yMacro) error {
		if haveOwner {
			return fmt.Errorf("duplicate owners macro: %s", m.Name)
		}

		if len(m.Args) > 0 {
			ya.Owners = m.Args
		}

		haveOwner = true
		return nil
	}

	for !haveDeclaration || !haveVersion || !haveOwner {
		m, err := tok.readMacro()
		if err != nil {
			return nil, err
		}

		if m == nil {
			break
		}

		switch m.Name {
		case MacroVersion:
			err = parseVersion(m)
		case MacroProgram:
			err = parseDeclaration(m, TargetGenericProgram)
		case MacroLibrary:
			err = parseDeclaration(m, TargetGenericLibrary)
		case MacroUDF, MacroYqlUDF:
			err = parseDeclaration(m, TargetGenericUDF)
		case MacroDLL:
			err = parseDeclaration(m, TargetGenericDLL)
		case MacroGoProgram:
			err = parseDeclaration(m, TargetGoProgram)
		case MacroGoLibrary:
			err = parseDeclaration(m, TargetGoLibrary)
		case MacroGoTest:
			err = parseDeclaration(m, TargetGoTest)
		case MacroGoDLL:
			err = parseDeclaration(m, TargetGoDLL)
		case MacroPyProgram, MacroPy3Program, MacroPy23Program, MacroPy2Program:
			err = parseDeclaration(m, TargetPyProgram)
		case MacroPyLibrary, MacroPy3Library, MacroPy23Library, MacroPy2Library, MacroPyModule, MacroPy2Module:
			err = parseDeclaration(m, TargetPyLibrary)
		case MacroPython3UDF, MacroPythonUDF:
			err = parseDeclaration(m, TargetPythonUDF)
		case MacroJavaProgram:
			err = parseDeclaration(m, TargetJavaProgram)
		case MacroJavaLibrary, MacroJavaContrib, MacroJavaContribProxy:
			err = parseDeclaration(m, TargetJavaLibrary)
		case MacroOwner:
			err = parseOwner(m)
		}

		if err != nil {
			return nil, err
		}
	}

	return ya, nil
}
