package commands

import (
	"context"
	"fmt"
	"os"
	"path"
	"path/filepath"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/mac-signer/internal/storage"
)

var dequeueArgs struct {
	Out    string
	Filter string
}

var dequeueCmd = &cobra.Command{
	Use:          "dequeue",
	SilenceUsage: true,
	RunE: func(_ *cobra.Command, args []string) error {
		s3, err := storage.NewStorage(&s3Config)
		if err != nil {
			return fmt.Errorf("unable to create s3 client: %w", err)
		}

		files, err := s3.List(context.Background(), "in/")
		if err != nil {
			return fmt.Errorf("list failed: %w", err)
		}

		for _, f := range files {
			if ok, _ := path.Match(dequeueArgs.Filter, f); !ok {
				logger.Info("skip filtered file", log.String("name", f))
				continue
			}

			remotePath := path.Join("in", f)
			err := func() error {
				logger.Info("download file", log.String("name", f))
				dst, err := os.Create(filepath.Join(dequeueArgs.Out, f))
				if err != nil {
					return fmt.Errorf("unable to create destination path: %w", err)
				}
				defer func() { _ = dst.Close() }()

				return s3.Download(context.Background(), remotePath, dst)
			}()

			if err != nil {
				return fmt.Errorf("unable to download file %q: %w", f, err)
			}

			logger.Info("downloaded")
			logger.Info("done")
		}
		return nil
	},
}

func init() {
	flags := dequeueCmd.PersistentFlags()
	flags.StringVar(&dequeueArgs.Out, "out", ".", "out")
	flags.StringVar(&dequeueArgs.Filter, "filter", "*", "filter")
}
