package commands

import (
	"context"
	"errors"
	"fmt"
	"os"
	"path"
	"path/filepath"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/mac-signer/internal/storage"
)

var queueArgs struct {
	ToolName string
	Version  string
}

var queueCmd = &cobra.Command{
	Use:          "queue",
	SilenceUsage: true,
	RunE: func(_ *cobra.Command, args []string) error {
		if queueArgs.ToolName == "" {
			return errors.New("no --name provided")
		}

		if queueArgs.Version == "" {
			return errors.New("no --version provided")
		}

		if len(args) == 0 {
			return errors.New("no file provided")
		}

		s3, err := storage.NewStorage(&s3Config)
		if err != nil {
			return fmt.Errorf("unable to create s3 client: %w", err)
		}

		f, err := os.Open(args[0])
		if err != nil {
			return fmt.Errorf("unable to open file: %w", err)
		}
		defer func() { _ = f.Close() }()

		logger.Info("start uploading")
		targetName := fmt.Sprintf("%s_%s%s", queueArgs.ToolName, queueArgs.Version, filepath.Ext(args[0]))
		uri, err := s3.Upload(context.Background(), path.Join("in", targetName), f)
		if err != nil {
			return fmt.Errorf("upload failed: %w", err)
		}

		logger.Info("uploaded", log.String("uri", uri))
		return nil
	},
}

func init() {
	flags := queueCmd.PersistentFlags()
	flags.StringVar(&queueArgs.ToolName, "name", "", "tool name to queue")
	flags.StringVar(&queueArgs.Version, "version", "", "tool version")
}
