package commands

import (
	"bufio"
	"fmt"
	"os"
	"path/filepath"
	"strings"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/mac-signer/internal/gon"
	"a.yandex-team.ru/security/mac-signer/internal/process"
)

var signArgs struct {
	In                  string
	BeforeSign          string
	AfterSign           string
	UserName            string
	BundleID            string
	ApplicationIdentity string
}

func ask4confirm(what, filename string) bool {
	reader := bufio.NewReader(os.Stdin)
	fmt.Printf("%s %s (Y/n): ", what, filename)
	s, err := reader.ReadString('\n')
	if err != nil {
		panic(err)
	}

	s = strings.TrimSpace(s)
	return len(s) == 0 || strings.EqualFold(s, "y") || strings.EqualFold(s, "yes")
}

var signCmd = &cobra.Command{
	Use:          "sign",
	SilenceUsage: true,
	RunE: func(_ *cobra.Command, args []string) error {
		files, err := filepath.Glob(filepath.Join(signArgs.In, "*"))
		if err != nil {
			return fmt.Errorf("unable to list files in dir %q: %w", signArgs.In, err)
		}

		for _, f := range files {
			if ok := ask4confirm("sign", f); !ok {
				logger.Info("skip file", log.String("name", f))
				continue
			}

			src := f
			if signArgs.BeforeSign != "" {
				logger.Info("prepare file", log.String("name", f))
				newPath, err := process.Exec(signArgs.BeforeSign, f)
				if err != nil {
					return err
				}
				src = newPath
			}

			logger.Info("start gon for file", log.String("name", f), log.String("src", src))
			err = gon.Sign(&gon.SignParams{
				Source:              src,
				BundleID:            signArgs.BundleID,
				ApplicationIdentity: signArgs.ApplicationIdentity,
			})
			if err != nil {
				return fmt.Errorf("sign fail: %w", err)
			}

			if signArgs.AfterSign != "" {
				logger.Info("postprocess file", log.String("name", f), log.String("src", src))
				_, err := process.Exec(signArgs.AfterSign, src, f)
				if err != nil {
					return err
				}
			}

			logger.Info("done")
		}

		return nil
	},
}

func init() {
	flags := signCmd.PersistentFlags()
	flags.StringVar(&signArgs.In, "in", ".", "in")
	flags.StringVar(&signArgs.BeforeSign, "before-sign", "", "prepare command")
	flags.StringVar(&signArgs.AfterSign, "after-sign", "", "postprocess command")
	flags.StringVar(&signArgs.UserName, "username", ".", "apple username")
	flags.StringVar(&signArgs.BundleID, "bundle-id", ".", "apple bundle id")
	flags.StringVar(&signArgs.ApplicationIdentity, "application-identity", ".", "apple application identity")
}
