package commands

import (
	"context"
	"fmt"
	"os"
	"path"
	"path/filepath"

	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/mac-signer/internal/storage"
)

var ignoredFiles = map[string]struct{}{
	".DS_Store": {},
}

var uploadArgs struct {
	In string
}

var uploadCmd = &cobra.Command{
	Use:          "upload",
	SilenceUsage: true,
	RunE: func(_ *cobra.Command, args []string) error {
		files, err := filepath.Glob(filepath.Join(uploadArgs.In, "*"))
		if err != nil {
			return fmt.Errorf("unable to list files in dir %q: %w", uploadArgs.In, err)
		}

		s3, err := storage.NewStorage(&s3Config)
		if err != nil {
			return fmt.Errorf("unable to create s3 client: %w", err)
		}

		for _, f := range files {
			if _, ignore := ignoredFiles[filepath.Base(f)]; ignore {
				logger.Info("ignore file", log.String("name", f))
				continue
			}

			if ok := ask4confirm("upload", f); !ok {
				logger.Info("skip file", log.String("name", f))
				continue
			}

			err := func() error {
				src, err := os.Open(f)
				if err != nil {
					return fmt.Errorf("unable to open file: %w", err)
				}
				defer func() { _ = src.Close() }()

				logger.Info("start uploading")
				targetName := filepath.Base(f)
				uri, err := s3.Upload(context.Background(), path.Join("out", targetName), src)
				if err != nil {
					return fmt.Errorf("upload failed: %w", err)
				}

				_ = s3.Delete(context.Background(), path.Join("in", targetName))
				logger.Info("uploaded", log.String("uri", uri))
				return nil
			}()

			if err != nil {
				return fmt.Errorf("failed to upload file %q: %w", f, err)
			}
		}

		return nil
	},
}

func init() {
	flags := uploadCmd.PersistentFlags()
	flags.StringVar(&uploadArgs.In, "in", ".", "in")
}
