package server

import (
	"context"
	"crypto/hmac"
	"crypto/sha256"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"net/url"
	"os"
	"os/exec"
	"path"
	"path/filepath"
	"strings"

	"github.com/labstack/echo/v4"
	"github.com/labstack/echo/v4/middleware"

	"a.yandex-team.ru/library/go/core/buildinfo"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/security/meat/internal/config"
	"a.yandex-team.ru/security/meat/internal/hub"
)

type (
	Server struct {
		cfg         *config.Config
		clickersHub *hub.Hub
		tvm         tvm.Client
		blackbox    blackbox.Client
	}

	User struct {
		blackbox.User
		UserTicket string
	}
)

func NewServer(cfg *config.Config) (*Server, error) {
	if cfg.TVM.CacheDir != "" {
		if err := os.MkdirAll(cfg.TVM.CacheDir, 0o700); err != nil {
			return nil, fmt.Errorf("unable to create tmm dir: %w", err)
		}
	}

	tvmSettings := tvmauth.TvmAPISettings{
		SelfID:                     cfg.TVM.ClientID,
		ServiceTicketOptions:       tvmauth.NewAliasesOptions(cfg.TVM.ClientSecret, cfg.TVM.Destinations),
		DiskCacheDir:               cfg.TVM.CacheDir,
		BlackboxEnv:                &cfg.TVM.Env,
		FetchRolesForIdmSystemSlug: cfg.Role.Slug,
		DisableDefaultUIDCheck:     true,
	}

	tvmClient, err := tvmauth.NewAPIClient(tvmSettings, &nop.Logger{})
	if err != nil {
		return nil, err
	}

	bbClient, err := httpbb.NewIntranet(httpbb.WithTVM(tvmClient))
	if err != nil {
		return nil, err
	}

	srv := &Server{
		cfg:         cfg,
		clickersHub: hub.NewHub(),
		tvm:         tvmClient,
		blackbox:    bbClient,
	}

	// TODO(buglloc): close?
	// TODO(buglloc: WTF?
	go srv.clickersHub.Start()

	return srv, nil
}

func (s *Server) getUser(c echo.Context) (*User, error) {
	sessID, err := c.Cookie("Session_id")
	if err != nil {
		return nil, err
	}

	rsp, err := s.blackbox.SessionID(
		c.Request().Context(),
		blackbox.SessionIDRequest{
			SessionID:     sessID.Value,
			UserIP:        c.RealIP(),
			Host:          c.Request().Host,
			GetUserTicket: true,
		},
	)

	if err != nil {
		log.Printf("failed to check user session: %v\n", err)
		return nil, err
	}

	return &User{
		User:       rsp.User,
		UserTicket: rsp.UserTicket,
	}, nil
}

func (s *Server) isValidUser(ctx context.Context, user *User) bool {
	roles, err := s.tvm.GetRoles(ctx)
	if err != nil {
		log.Println("unable to get user roles")
		return false
	}

	ticket, err := s.tvm.CheckUserTicket(ctx, user.UserTicket)
	if err != nil {
		log.Printf("unable to check user ticket for user %q: %s", user.Login, err)
		return false
	}

	ok, err := roles.CheckUserRole(ticket, s.cfg.Role.Name, nil)
	if err != nil {
		log.Printf("unable to check user %q role: %v\n", user.Login, err)
		return false
	}

	return ok
}

func (s *Server) isValidToken(tokenHeader string, body []byte) bool {
	// sha256=c325e83425e207306de1c83e0ea9fe285095e387e7110ef65db75da58db88541
	sign := strings.SplitN(tokenHeader, "=", 2)
	if len(sign) != 2 {
		return false
	}

	algo, actualMac := strings.TrimSpace(sign[0]), strings.TrimSpace(sign[1])
	if algo == "" || actualMac == "" {
		return false
	}

	if algo != "sha256" {
		log.Println("Unexpected algo: " + algo)
		return false
	}

	mac := hmac.New(sha256.New, []byte(s.cfg.SignSecret))
	_, _ = mac.Write(body)
	expectedMAC := fmt.Sprintf("%x", mac.Sum(nil))
	return hmac.Equal([]byte(expectedMAC), []byte(actualMac))
}

func (s *Server) rebuild(wait bool) error {
	cmd := exec.Command(s.cfg.BuildCmd)
	cmd.Env = append(os.Environ(), "GIT_TERMINAL_PROMPT=0")
	if err := cmd.Start(); err != nil {
		return err
	}

	if wait {
		if err := cmd.Wait(); err != nil {
			return err
		}
	}
	return nil
}

func (s *Server) userAuthMiddleware(next echo.HandlerFunc) echo.HandlerFunc {
	return func(c echo.Context) error {
		user, err := s.getUser(c)
		if err != nil {
			return c.JSON(http.StatusUnauthorized, echo.Map{"status": "unauthorized"})
		}

		if !s.isValidUser(c.Request().Context(), user) {
			return c.JSON(http.StatusUnauthorized, echo.Map{
				"status":       "access denied",
				"user":         user.Login,
				"missing_role": s.cfg.Role.Name,
			})
		}

		return next(c)
	}
}

func (s *Server) ListenAndServe() error {
	e := echo.New()
	e.Debug = s.cfg.Debug
	e.Use(middleware.Recover())
	e.Use(middleware.Logger())

	e.File("/favicon.ico", "./static/favicon.ico")
	e.GET("/ping", func(c echo.Context) error {
		return c.String(200, "pong")
	})

	e.Static("/static", "./static")

	e.GET("/cl", func(c echo.Context) error {
		return c.File("./static/clicker.html")
	})

	e.GET("/cl/ws", s.clickerWsHandler, s.userAuthMiddleware)

	e.GET("/version", func(c echo.Context) error {
		return c.String(200, buildinfo.Info.ProgramVersion)
	})

	e.GET("/roles", func(c echo.Context) error {
		roles, err := s.tvm.GetRoles(c.Request().Context())
		if err != nil {
			return fmt.Errorf("unale to get TVM roles: %w", err)
		}

		return c.JSON(http.StatusOK, roles.GetMeta())
	})

	e.POST("/api/hook", func(c echo.Context) (err error) {
		if c.Request().Header.Get("X-Event-Key") != "repo:refs_changed" {
			return c.String(http.StatusOK, "ok")
		}

		tokenHeader := c.Request().Header.Get("X-Hub-Signature")
		if tokenHeader == "" {
			return c.JSON(http.StatusBadRequest, echo.Map{"error": "no auth header"})
		}

		body, err := ioutil.ReadAll(c.Request().Body)
		if err != nil {
			return c.JSON(http.StatusBadRequest, echo.Map{"error": err.Error()})
		}

		if !s.isValidToken(tokenHeader, body) {
			return c.JSON(http.StatusForbidden, echo.Map{"status": "access denied"})
		}

		var hook struct {
			Changes []struct {
				RefID string `json:"refId"`
			} `json:"changes"`
		}

		if err = json.Unmarshal(body, &hook); err != nil {
			return c.JSON(http.StatusBadRequest, echo.Map{"error": err.Error()})
		}

		masterPush := false
		for _, change := range hook.Changes {
			if change.RefID == "refs/heads/master" {
				masterPush = true
				break
			}
		}

		if !masterPush {
			// Skip push to not master branch
			return c.JSON(http.StatusOK, echo.Map{"changes": hook.Changes})
		}

		if err = s.rebuild(false); err != nil {
			return c.JSON(http.StatusBadRequest, echo.Map{"error": err.Error()})
		}

		return c.JSON(http.StatusOK, echo.Map{"changes": hook.Changes})
	})

	e.GET("/api/rebuild", func(c echo.Context) (err error) {
		if err = s.rebuild(true); err != nil {
			return c.JSON(http.StatusBadRequest, echo.Map{"error": err.Error()})
		}

		return c.JSON(http.StatusOK, echo.Map{"status": "ok"})
	}, s.userAuthMiddleware)

	e.GET("/*", func(c echo.Context) error {
		p, err := url.PathUnescape(c.Param("*"))
		if err != nil {
			return err
		}

		name := filepath.Join(s.cfg.StaticRoot, path.Clean("/"+p)) // "/"+ for security
		return c.File(name)
	}, s.userAuthMiddleware)

	return e.Start(fmt.Sprintf(":%d", s.cfg.Port))
}
