package main

import (
	"context"
	"encoding/json"
	"flag"
	"fmt"
	"io"
	"io/ioutil"
	"log"
	"net/http"
	"net/url"
	"strings"
	"time"

	"github.com/gofrs/uuid"
	zp "go.uber.org/zap"
	"go.uber.org/zap/zapcore"

	"a.yandex-team.ru/library/go/core/log/zap"
)

type Header struct {
	Name  string
	Value []string
}

type SerpHeader struct {
	Data []string
}

type SerpReq struct {
	Method  string // GET/POST/etc
	Path    string `json:"request"`
	Proto   string `json:"protocol"` // "HTTP/1.0"
	Headers []*SerpHeader
}

type Request struct {
	UnixTime  int64 `json:"unixtime"`
	UniqId    string
	ServiceId string
	Method    string // GET/POST/etc
	Path      string
	RawQuery  string
	Proto     string // "HTTP/1.0"
	Headers   []*Header
	Body      []byte
	Date      time.Time
}

type AppContext struct {
	lbWriter LbWriter
	conf     Config
}

type Config struct {
	Bind             string   `json:"bind"`
	PushClientPath   []string `json:"push_client"`
	SensitiveCookies []string `json:"sensitive_cookies"`
	SensitiveHeaders []string `json:"sensitive_headers"`
}

func maskCookies(lines []string, sensitiveCookies []string) []string {
	cookies := make([]string, 0)
	for _, line := range lines {
		parts := strings.Split(strings.TrimSpace(line), ";")

		if len(parts) == 1 && parts[0] == "" {
			return []string{""}
		}

		for i := 0; i < len(parts); i++ {
			parts[i] = strings.TrimSpace(parts[i])

			if len(parts[i]) == 0 {
				continue
			}

			name, val := parts[i], ""

			if j := strings.Index(name, "="); j >= 0 {
				name, val = name[:j], name[j+1:]
			}

			var sensitive = false
			for _, cn := range sensitiveCookies {
				if name == cn {
					sensitive = true
					break
				}
			}

			if sensitive {
				idx := strings.LastIndex(val, ".")
				if idx > -1 {
					val = val[:idx] + ".censored"
				}
			}
			cookies = append(cookies, name+"="+val)
		}
	}
	return []string{strings.TrimSpace(strings.Join(cookies[:], "; "))}
}

func (ctx *AppContext) HandlePingReq(w http.ResponseWriter, r *http.Request) {
	_, _ = fmt.Fprintf(w, "ok")
}

func (ctx *AppContext) HandleReq(w http.ResponseWriter, r *http.Request) {
	_, _ = fmt.Fprintf(w, "ok")
	raw := new(Request)
	raw.UnixTime = time.Now().Unix()
	uuid, _ := uuid.NewV4()
	raw.UniqId = uuid.String()
	for key, value := range r.Header {
		h := new(Header)
		h.Name = key
		if key == "X-Antirobot-Service-Y" {
			for _, ServiceId := range value {
				raw.ServiceId = ServiceId
				break
			}
		}
		if key == "Cookie" {
			value = maskCookies(value, ctx.conf.SensitiveCookies)
		}
		for _, shn := range ctx.conf.SensitiveHeaders {
			if shn == key {
				value = []string{"censored"}
				break
			}
		}
		h.Value = value
		raw.Headers = append(raw.Headers, h)
	}

	if raw.ServiceId == "serp-priemka" {
		if r.Body == nil {
			return
		}

		var sr SerpReq

		var reader io.Reader = r.Body
		body, _ := ioutil.ReadAll(reader)
		err := json.Unmarshal(body, &sr)
		if err != nil {
			return
		}

		u, err := url.ParseRequestURI(sr.Path)
		if err != nil {
			return
		}

		raw.Method = sr.Method
		raw.Path = u.Path
		raw.RawQuery = u.RawQuery
		raw.Proto = r.Proto
		for _, value := range sr.Headers {
			if len(value.Data) < 2 {
				continue
			}
			if value.Data[0] == "Cookie" {
				value.Data[1] = maskCookies([]string{value.Data[1]}, ctx.conf.SensitiveCookies)[0]
				continue
			}
			h := new(Header)
			h.Name = value.Data[0]
			h.Value = append(h.Value, value.Data[1])
			raw.Headers = append(raw.Headers, h)
		}

	} else {
		raw.Method = r.Method
		raw.Path = r.URL.Path
		raw.RawQuery = r.URL.RawQuery
		raw.Proto = r.Proto

		if r.Body != nil {
			var reader io.Reader = r.Body
			raw.Body, _ = ioutil.ReadAll(reader)
		}
	}

	b, _ := json.Marshal(raw)

	ctx.lbWriter.Write(b)
	ctx.lbWriter.Write([]byte("\n"))
}

func (ctx *AppContext) loadConfig(fileName string) *Config {
	var conf Config

	file, err := ioutil.ReadFile(fileName)
	if err != nil {
		log.Fatal(err)
		return nil
	}

	err = json.Unmarshal(file, &conf)
	if err != nil {
		log.Fatal(err)
		return nil
	}

	ctx.conf = conf
	return &conf
}

func (ctx *AppContext) initLbWriter() {
	logger := zap.Must(zp.Config{
		Level:            zp.NewAtomicLevelAt(zp.InfoLevel),
		Encoding:         "console",
		OutputPaths:      []string{"stdout"},
		ErrorOutputPaths: []string{"stderr"},
		EncoderConfig: zapcore.EncoderConfig{
			MessageKey:     "msg",
			LevelKey:       "level",
			TimeKey:        "ts",
			CallerKey:      "caller",
			EncodeLevel:    zapcore.CapitalColorLevelEncoder,
			EncodeTime:     zapcore.ISO8601TimeEncoder,
			EncodeDuration: zapcore.StringDurationEncoder,
			EncodeCaller:   zapcore.ShortCallerEncoder,
		},
	})
	lbCtx := context.Background()

	lbWriter := NewLbWriter(logger, lbCtx)
	ctx.lbWriter = lbWriter
}

func main() {
	ctx := &AppContext{}
	var configFile = flag.String("config", "./config.json", "Config file path.")
	flag.Parse()

	log.Printf("Parsing config")
	cfg := ctx.loadConfig(*configFile)
	if cfg == nil {
		log.Fatal("Invalid config ... exiting")
		return
	}
	log.Printf("Starting lbWriter")
	ctx.initLbWriter()

	log.Printf("Starting HTTP listener")
	// producer
	http.HandleFunc("/", ctx.HandleReq)
	http.HandleFunc("/ping", ctx.HandlePingReq)
	log.Fatal(http.ListenAndServe(ctx.conf.Bind, nil))
}
