# -*- coding: utf-8 -*-

from logging import getLogger

from datetime import datetime
from dateutil.relativedelta import relativedelta

from django.contrib import admin
from django.utils.safestring import mark_safe
from django.utils.translation import ugettext_lazy as _
from django.conf import settings
from django.contrib import messages
from django.contrib.admin.actions import delete_selected

from .models import ScanProfile, Vulnerability, VulnerabilityType, Target, \
    ScanTask, Scan, ProfilesTargets, VulnTicket, FalsePositive, HTTPTransaction, \
    AuthProfile, FalsePositiveByVulnType, CrasherStatus, RequestSamples, TargetUriMap
from .tasks import stop_remote_scan

logger = getLogger(__name__)


class ScanProfileAdmin(admin.ModelAdmin):
    list_display = ['name', 'short_comment', 'is_public']
    search_fields = ['name', 'short_comment']
    list_filter = ['is_public']
    ordering = ('-id',)


class ScanAdmin(admin.ModelAdmin):
    list_filter = ['status', 'is_prod']
    search_fields = ['url', 'scan_task__name', 'scan_task__comment', 'pidfile', 'task_id', 'uid']
    list_display = ['status', 'url', 'start', 'finish', 'is_prod', 'target', 'agent_host']
    ordering = ('-id',)
    actions = ['stop_action', 'delete_selected']
    readonly_fields = ('scan_task',)
    exclude = ('sample_requests',)


    def delete_selected(self, request, queryset):
        for obj in queryset:
            if obj.status == Scan.ST_INPROGRESS:
                stop_remote_scan.delay(obj.id)
        return delete_selected(self, request, queryset)

    delete_selected.short_description = _('Delete selected %(verbose_name_plural)s')

    def stop_action(self, request, queryset):
        for selected_obj in queryset:
            selected_obj.status = Scan.ST_ABORTED
            selected_obj.save()
            stop_remote_scan.delay(selected_obj.id)
        messages.success(request, _('Scans stoped successfully.'))

    stop_action.short_description = _('Stop selected %(verbose_name_plural)s')

    def get_target(self, obj):
        return mark_safe(obj.scan_task.target)

    get_target.short_description = _('Target')

    def comment(self, obj):
        return mark_safe(obj.scan_task.comment)

    comment.short_description = _('Description')

    def scan_task_link(self, obj):
        return mark_safe('<a href="../scantask/%s/"> task </a>' % (
                         obj.scan_task_id
                        ))

    scan_task_link.short_description = _('Task name')
    scan_task_link.allow_tags = True
    scan_task_link.admin_order_field = 'scan_task__name'

    def icon(self, obj):
        icons_status = {
           Scan.ST_INPROGRESS: {'icon': 'icon-in-proc.gif',
                                'hint': _('In process')},
           Scan.ST_DONE: {'icon': 'icon-yes.gif',
                          'hint': _('Finish successfull')},
           Scan.ST_FAIL: {'icon': 'icon-no.gif',
                          'hint': _('Fail')},
           Scan.ST_ABORTED: {'icon': 'icon-no.gif',
                             'hint': _('Aborted')},
           'done_and_vulnerable': {'icon': 'icon-yes-vuln.gif',
                                   'hint': _('Finished successfull and vulnerabilies found')},
        }
        status = obj.status
        vuln_count = Vulnerability.objects.filter(scan=obj).exists()
        if vuln_count and obj.status == Scan.ST_DONE:
            status = 'done_and_vulnerable'
        icon = icons_status.get(status)['icon']
        hint = icons_status.get(status)['hint']
        return mark_safe(
            '<img src="%s/molly_webui/icons/%s" alt="%s" title="%s" />' % (settings.STATIC_URL,
                                                                            icon,
                                                                            hint,
                                                                            hint,
                                                                            ))

    icon.short_description = _('S')
    icon.allow_tags = True
    icon.admin_order_field = 'status'

    def show_report(self, obj):
        return mark_safe('<a href="/report/%s/">%s</a>' % (obj.uid, _('Show report')))

    show_report.short_description = _('Report')
    show_report.allow_tags = True

    def queryset(self, request):
        latest_old_scan_time = datetime.now() + relativedelta(days=-60)
        return Scan.objects.filter(start__gte=latest_old_scan_time)

    def changelist_view(self, request, extra_context=None):
        extra_context = {'title': '%s' % _('Scans'), }
        return super(ScanAdmin, self).changelist_view(request, extra_context)


class ScanTaskAdmin(admin.ModelAdmin):
    """
    Class for view scans in admin interface
    """
    list_display = ['name', 'target_name', 'comment', 'get_report']
    ordering = ('-id',)
    actions = ['delete_selected']
    search_fields = ['name', 'comment', 'target__name', 'target__url']

    def target_name(self, obj):
        return mark_safe('<a href="/target/%s/">%s</a>' % (obj.target.id, obj.target.name))

    target_name.short_description = _('target')
    target_name.allow_tags = True
    target_name.admin_order_field = 'target__name'

    def get_report(self, obj):
        try:
            scan = Scan.objects.filter(scan_task=obj.id).order_by('-start')[0]
            return mark_safe('<a href="/report/%s/">%s</a>' % (scan.uid, scan.start))
        except Exception:
            pass
        return ''

    get_report.short_description = _('Last scan')
    get_report.allow_tags = True

    def delete_selected(self, request, queryset):
        return delete_selected(self, request, queryset)

    delete_selected.short_description = _('Delete selected %(verbose_name_plural)s')


class TargetAdmin(admin.ModelAdmin):
    list_display = ['name', 'url', 'st_queue', 'parent', 'last_scan', 'get_stat_link']
    search_fields = ['name', 'url', 'st_queue', 'antirobot_uid']
    list_filter = ['st_queue', 'abc_id']
    ordering = ('-id',)
    raw_id_fields = ('parent',)

    def get_profiles(self, obj):
        all_profiles = ProfilesTargets.objects.filter(target=obj)
        profile_names = [x.scan_profile.name for x in all_profiles]
        return ','.join(profile_names)

    get_profiles.short_description = _('Default scan profiles')
    get_profiles.allow_tags = True

    def get_stat_link(self, obj):
        return mark_safe('<a href="../target_stats/%s/"> %s </a>' % (obj.id,
                                                                      _('Show stats')
                                                                      ))

    get_stat_link.short_description = _('Stats')
    get_stat_link.allow_tags = True


class VulnerabilityTypeAdmin(admin.ModelAdmin):
    list_display = ['name', 'short_summary', 'description_url', 'severity',
                    'scanner_severity', 'tracker_severity', 'combine', 'is_internal', 'archived',
                    'st_tags']
    list_filter = ['archived', 'severity', 'combine', 'is_internal', 'tracker_severity']


class VulnerabilityAdmin(admin.ModelAdmin):
    list_display = ['id', 'vuln_type']
    list_filter = ['vuln_type']
    readonly_fields = ("http_details", 'scan')


class VulnTicketAdmin(admin.ModelAdmin):
    list_display = ['ticket_id']


class FalsePositiveAdmin(admin.ModelAdmin):
    list_display = ['target', 'vuln_type']


class AuthProfileAdmin(admin.ModelAdmin):
    search_fields = ['uid', 'username']
    list_display = ['uid', 'username']


class FalsePositiveByVulnTypeAdmin(admin.ModelAdmin):
    list_display = ['vuln_type', 'target', 'deleted']


class CrasherStatusAdmin(admin.ModelAdmin):
    list_display = ['status', 'user', 'last_updated']


class RequestSamplesAdmin(admin.ModelAdmin):
    list_display = ['format', 'uid', 'url']
    search_fields = ['uid']
    list_filter = ['format']


class TargetUriMapAdmin(admin.ModelAdmin):
    list_display = ['uid']


class HTTPTransactionAdmin(admin.ModelAdmin):
    list_display = ['request_line']
    readonly_fields = ('request_headers', 'response_headers')


admin.site.register(Target, TargetAdmin)
admin.site.register(ScanTask, ScanTaskAdmin)
admin.site.register(Scan, ScanAdmin)
admin.site.register(VulnerabilityType, VulnerabilityTypeAdmin)
admin.site.register(FalsePositiveByVulnType, FalsePositiveByVulnTypeAdmin)
admin.site.register(ScanProfile, ScanProfileAdmin)
admin.site.register(Vulnerability, VulnerabilityAdmin)
admin.site.register(VulnTicket, VulnTicketAdmin)
admin.site.register(FalsePositive, FalsePositiveAdmin)
admin.site.register(AuthProfile, AuthProfileAdmin)
admin.site.register(CrasherStatus, CrasherStatusAdmin)
admin.site.register(RequestSamples, RequestSamplesAdmin)
admin.site.register(TargetUriMap, TargetUriMapAdmin)
admin.site.register(HTTPTransaction, HTTPTransactionAdmin)
