import logging

from django.conf import settings
from django.contrib import auth

from rest_framework.authentication import get_authorization_header, BaseAuthentication
from rest_framework import exceptions, HTTP_HEADER_ENCODING
import datetime

from django.contrib.auth.models import User
from django_yauth.util import get_real_ip, current_host

from molly_webui.utils import mask_credentials

LOG = logging.getLogger(__name__)


class YandexTokenAuthentication(BaseAuthentication):
    """
    Clients should authenticate by passing the token key in the "Authorization"
    HTTP header, prepended with the string "OAuth ".  For example:

        Authorization: OAuth 401f7ac837da42b97f613d789819ff93537bee6a
    """

    def authenticate(self, request):
        auth_info = get_authorization_header(request).split()

        if not auth_info or auth_info[0].lower() not in [b'oauth', b'static']:
            request.__class__.yauser = None
            raise exceptions.AuthenticationFailed('Not authorized')

        if len(auth_info) == 1:
            msg = 'Invalid token header. No credentials provided.'
            raise exceptions.AuthenticationFailed(msg)
        elif len(auth_info) > 2:
            msg = 'Invalid token header. Token string should not contain spaces.'
            raise exceptions.AuthenticationFailed(msg)

        if auth_info[1] in settings.TOKENS_BLACKLIST:
            msg = 'Token blacklisted. Get a new one.'
            raise exceptions.AuthenticationFailed(msg)

        if auth_info[1] in settings.STATIC_TOKENS:
            auth_info[1] = settings.OAUTH_TOKEN_FOR_STATIC

        try:
            token = auth_info[1].decode("latin-1")
            session = self._get_session(token, get_real_ip(request), current_host(request))
            if session.get('status', '') != 'VALID':
                LOG.critical('auth_error="%s" ip="%s" status="%s"' % (mask_credentials(auth_info[1]),
                                                                      get_real_ip(request),
                                                                      session.get('status', '')))
                raise exceptions.AuthenticationFailed('Invalid session')

            fields = session.get('fields', dict())
            if not [perm for perm in session.get('oauth', dict()).get('scope', '').split(' ')
                    if perm in settings.OAUTH_VALID_SCOPE]:
                msg = 'Token does not have valid scope.'
                raise exceptions.AuthenticationFailed(msg)

            if fields and not fields.get('username'):
                fields['username'] = fields.get('login')

            login = fields.get('username')
            if not login:
                LOG.critical('auth_error="%s" ip="%s"' % (mask_credentials(auth_info[1]),
                                                          get_real_ip(request)))
                raise exceptions.AuthenticationFailed('Passport error')

            user = self.get_user(login)
            user.backend = 'django.contrib.auth.backends.RemoteUserBackend'
            return (user, None)
        except Exception as err:
            LOG.critical('auth_exception="%s" ip="%s" err="%s"' % (mask_credentials(auth_info[1]),
                                                                   get_real_ip(request),
                                                                   str(err).strip()))
            raise exceptions.AuthenticationFailed('Invalid token')

        LOG.critical('auth_error="%s" ip="%s"' % (mask_credentials(auth_info[1]), get_real_ip(request)))

        raise exceptions.AuthenticationFailed('Not authorized')


    def _get_session(self, key, ip, host):
        bb = settings.YAUTH_BLACKBOX_INSTANCE
        try:
            session = bb.oauth(key, ip, dbfields=list(settings.PASSPORT_FIELDS), by_token=True)
        except Exception:
            raise
        return session

    def authenticate_header(self, request):
        return 'Static'

    def get_user(self, login):
        try:
            user = User.objects.get(username=login)
        except User.DoesNotExist:
            user = User(username=login, email=login+'@yandex-team.ru', last_login=datetime.datetime.now())
            user.save()
        return user
