# -*- coding: utf-8 -*-
import datetime
import json
import hashlib
import urllib.parse
import memcache
import operator
import logging
import zlib
import traceback 

from dateutil.relativedelta import relativedelta

from django.contrib.auth.models import User
from django.core.validators import URLValidator
from django.core.exceptions import ValidationError
from django.utils.decorators import method_decorator
from django.views.decorators.csrf import csrf_exempt
from django.shortcuts import get_object_or_404
from django.conf import settings
from django.urls import reverse
from django.db.models import Q

from rest_framework.views import APIView
from rest_framework.response import Response
from rest_framework import status
from rest_framework.parsers import FormParser, MultiPartParser, JSONParser

from molly_webui.models import Scan, ScanTask, Target, ProfilesTasks, ProfilesTargets, ScanProfile,\
    Vulnerability, VulnTicket, AuthProfile, FalsePositive, VulnerabilityType, CrasherStatus, TargetUriMap, \
    RequestSamples, CrasherChunk

from molly_webui.tasks import run_scan
from molly_webui.utils import upload_to_elliptics, join_by_domain, slugify_target_url, get_abc_id_by_slug, \
    get_abc_id_by_url, get_latest_aggregate, get_cgroup, get_abc_members, clean_antirobot_header

from startrek_client import Startrek
from molly_webui.notification.errors import catch_error

from .authentication import YandexTokenAuthentication
from .permissions import IsYandexUser, IsSuperUser, IsStaffUser
from functools import reduce

LOG = logging.getLogger(__name__)


class StatHandler(APIView):
    authentication_classes = (YandexTokenAuthentication,)
    permission_classes = (IsSuperUser,)

    def get_name(self):
        return 'Molly statistics'

    def get(self, request, *args, **kwargs):
        latest_old_scan_time = datetime.datetime.now() + relativedelta(days=-1)
        total_targets = Target.objects.filter(parent__isnull=True).count()
        total_scans = Scan.objects.filter(start__gte=latest_old_scan_time, is_prod=False).count()
        crasher_scans = Scan.objects.filter(start__gte=latest_old_scan_time, is_prod=True).count()
        vulns = Vulnerability.objects.filter(scan__start__gte=latest_old_scan_time,
                                             scan__is_prod=False)
        total_vulns = 0
        triaged_vulns = 0
        for item in vulns:
            if not item.is_triaged and not item.is_false_positive:
                total_vulns += 1
            elif item.is_triaged:
                triaged_vulns += 1
        nonprod_targets = 0
        targets = Target.objects.filter(parent__isnull=True)
        for t in targets:
            if t.is_prod:
                continue
            nonprod_targets += 1
        active_tickets = VulnTicket.objects.filter(last_updated__gte=latest_old_scan_time).count()
        false_positives = FalsePositive.objects.filter(last_updated__gte=latest_old_scan_time).count()
        active_scans = Scan.objects.filter(start__gte=latest_old_scan_time,
                                           status=Scan.ST_INPROGRESS).count()
        aborted_scans = Scan.objects.filter(start__gte=latest_old_scan_time,
                                           status=Scan.ST_ABORTED).count()
        return Response({'targets': total_targets, 'total_scans': total_scans,
                         'total_vulns': total_vulns, 'active_tickets': active_tickets,
                         'false_positives': false_positives, 'active_scans': active_scans,
                         'aborted_scans': aborted_scans, 'crasher_scans': crasher_scans,
                         'triaged_vulns': triaged_vulns, 'molly_targets': nonprod_targets})


class TargetList(APIView):
    authentication_classes = (YandexTokenAuthentication,)
    permission_classes = (IsStaffUser,)

    def get_name(self):
        return 'Molly target list api'

    def get(self, request, *args, **kwargs):
        url = request.GET.get('url', '')
        slug = slugify_target_url(url)
        st_queue = request.GET.get('tracker_startrek', '')
        abc_id = request.GET.get('abc_id', '')
        targets = []
        if abc_id:
            qs = Target.objects.filter(abc_id=abc_id).order_by('-last_scan')
        elif st_queue:
            qs = Target.objects.filter(st_queue=st_queue).order_by('-last_scan')
        elif url and slug:
            known = set()
            qs = Target.objects.filter(slug=slug).order_by('-last_scan')
            for t in qs:
                if t.parent:
                    t = t.parent
                if t.id in known:
                    continue
                targets.append({'name': t.name,
                                'last_scan': t.last_scan,
                                'last_scan_vulns': t.last_scan_vulns,
                                'tracker_startrek': t.st_queue,
                                'abc_id': t.abc_id,
                                'url': t.url,
                                'slug': t.slug,
                                'is_prod': t.is_prod,
                                'antirobot_uid': t.antirobot_uid})
                known.add(t.id)
            qs = Scan.objects.filter(slug=slug).order_by('-id').exclude(target__in=list(known)).values("target_id").distinct()
            for tid in qs:
                if tid.get('target_id') in known:
                    continue
                try:
                    t = Target.objects.get(id=tid.get('target_id'))
                except Exception:
                    catch_error()
                    continue
                else:
                    if t.parent:
                        t = t.parent
                    if t.id in known:
                        continue
                    targets.append({'name': t.name,
                                    'last_scan': t.last_scan,
                                    'last_scan_vulns': t.last_scan_vulns,
                                    'tracker_startrek': t.st_queue,
                                    'abc_id': t.abc_id,
                                    'url': t.url,
                                    'is_prod': t.is_prod,
                                    'slug': t.slug,
                                    'antirobot_uid': t.antirobot_uid})
                    known.add(t.id)
            if not targets:
                ptu = urllib.parse.urlparse(url)
                cgroup = get_cgroup(ptu.netloc)
                if not ('slb' in cgroup or 'qloud-b' in cgroup or 'ipvsfwmark' in cgroup):
                    filters = [Q(name='{}'.format(cgroup)), Q(name='CRASHER_{}'.format(ptu.netloc))]
                    qs = Target.objects.filter(reduce(operator.or_, filters)).order_by('-last_scan')
                    for t in qs:
                        if t.parent:
                            t = t.parent
                        if t.id in known:
                            continue
                        targets.append({'name': t.name,
                                        'last_scan': t.last_scan,
                                        'last_scan_vulns': t.last_scan_vulns,
                                        'tracker_startrek': t.st_queue,
                                        'abc_id': t.abc_id,
                                        'url': t.url,
                                        'slug': t.slug,
                                        'is_prod': t.is_prod,
                                        'antirobot_uid': t.antirobot_uid})
                        known.add(t.id)
            return Response({'targets': targets})
        else:
            qs = Target.objects.filter(parent__isnull=True).order_by('-last_scan')
        for t in qs:
            targets.append({'id': t.id,
                            'name': t.name,
                            'last_scan': t.last_scan,
                            'last_scan_vulns': t.last_scan_vulns,
                            'tracker_startrek': t.st_queue,
                            'abc_id': t.abc_id,
                            'slug': t.slug,
                            'url': t.url,
                            'is_prod': t.is_prod,
                            'antirobot_uid': t.antirobot_uid})
        return Response({'targets': targets})


class TargetHandler(APIView):
    def get_name(self):
        return 'Molly1.1 targets resolve api'

    def target_details(self, target):
        return Response({'target': {'name': target.name,
                                    'id': target.id,
                                    'tracker_startrek': target.st_queue,
                                    'abc_id': target.abc_id}})

    def get(self, request, *args, **kwargs):
        target = get_object_or_404(Target, id=kwargs.get('id'))
        return self.target_details(target)


class APIError(Exception):
    def __init__(self, value, status_code):
        self.value = value
        if status_code:
            self.status_code = status_code
        else:
            self.status_code = status.HTTP_400_BAD_REQUEST

    def __str__(self):
        return self.value, self.status_code


class ScanHandler(APIView):
    authentication_classes = (YandexTokenAuthentication,)
    permission_classes = (IsYandexUser,)
    parser_classes = (FormParser,)

    def get_name(self):
        return 'Molly Scan API v.1.1'

    def get(self, request, *args, **kwargs):
        if not kwargs.get('scan_id'):
            return Response({'error': 'no scan_id specified'},
                            status=status.HTTP_400_BAD_REQUEST)

        scan = get_object_or_404(Scan, uid=kwargs.get('scan_id'))
        if scan.status == Scan.ST_DONE:
            resp = dict()
            resp['status'] = 'done'
            resp['vulnerabilities'] = scan.vulnerabilities_list
            resp['report_url'] = settings.APP_URL + reverse('show_report', args=[scan.uid])
            return Response(resp)
        elif scan.status == Scan.ST_ABORTED:
            resp = dict()
            resp['status'] = 'aborted'
            resp['vulnerabilities'] = scan.vulnerabilities_list
            resp['report_url'] = settings.APP_URL + reverse('show_report', args=[scan.uid])
            return Response(resp)
        elif scan.status == Scan.ST_FAIL:
            resp = dict()
            resp['error'] = scan.result_message
            resp['status'] = 'error'
            resp['vulnerabilities'] = scan.vulnerabilities_list
            resp['report_url'] = settings.APP_URL + reverse('show_report', args=[scan.uid])
            return Response(resp)
        return Response({'status': 'in_progress'})

    def post(self, request, *args, **kwargs):
        """Create and run scan. Return scan_id of created scan.

        There are three possible combinations of parameters.
          1. target is set and corresponding model Target has
        ProfilesTargets configured.
          2. target and profile are set.
          3. profile is set and target_uri is set and points to
        existing and accessible location.

        Accepted arguments:
        :target:  name of predefined target
        :profile:  names of predefined profiles (can be more than one)
        :target_uri:  custom target_uri (requires profile)
        """

        target_name = request.POST.get('target')
        profiles = request.POST.getlist('profile')
        target_uris = request.POST.getlist('target_uri')
        st_queue = request.POST.get('st_queue')
        usernames = request.POST.get('users')
        abc_id = request.POST.get('abc_id', 0)
        user_agent = request.POST.get('user_agent', '')
        auth_profile = request.POST.get('auth_profile', settings.DEFAULT_AUTH_PROFILE)
        request_samples_uid = request.POST.get('request_samples', '')
        is_prod = request.POST.get('is_prod', False)
        ignore_time_limit = request.POST.get('ignore_time_limit', False)
        qs_params = request.POST.get('qs_params', '')
        st_ticket = request.POST.get('st_ticket')
        send_mail = request.POST.get('send_mail', False)
        skip_ongoing_check = request.POST.get('skip_ongoing_check', False)
        rps = int(request.POST.get('rps', -1))
        severity = request.POST.get('severity', VulnerabilityType.SEVERITY_MEDIUM)
        no_auto_tickets = request.POST.get('no_auto_tickets', False)
        target_map_uid = request.POST.get('target_map', '')
        request_samples_resource = request.POST.get('request_samples_resource', '')
        request_samples_resource_type = request.POST.get('request_samples_resource_type', RequestSamples.FMT_SERP)
        request_samples_aggregate_uid = request.POST.get('request_samples_aggregate_uid', '')
        collaborator_type = request.POST.get('collaborator', Scan.COLLAB_TYPE_PRIVATE)
        scanner_type = int(request.POST.get("scanner_type", 1))

        scanlist = []
        cl_uris = join_by_domain(target_uris)
        for k in list(cl_uris.keys()):
            urls = cl_uris.get(k)[::-1]
            target_uri = urls.pop()
            try:
                res, new = self.create_scan(k + target_uri, auth_profile, user_agent, urls, request,
                                            target_name, profiles, st_queue, usernames, target_map_uid,
                                            is_prod, ignore_time_limit, qs_params, st_ticket, send_mail, abc_id,
                                            request_samples_uid, rps, severity, no_auto_tickets,
                                            request_samples_resource, request_samples_resource_type,
                                            request_samples_aggregate_uid, collaborator_type, scanner_type, skip_ongoing_check)
            except APIError as err:
                return Response({'status': 'error', 'error': err.value}, status=err.status_code)
            else:
                if res:
                    scanlist.append((res, new))
        if not scanlist:
            return Response({'status': 'error', 'error': 'No scans created. Please check params.'},
                            status=status.HTTP_400_BAD_REQUEST)
        ret = []
        for scan, new in scanlist:
            if new:
                run_scan.delay(scan.id)
            ret.append(scan.uid)
        return Response({'status': 'in_progress', 'scan_id': ret})

    def create_scan(self, target_uri, auth_profile_uid, user_agent, urls, request, target_name, profiles,
                    st_queue, usernames, target_map_uid, is_prod, ignore_time_limit, qs_params,
                    st_ticket, send_mail, abc_id, request_samples_uid, rps, severity, no_auto_tickets,
                    request_samples_resource, request_samples_resource_type, request_samples_aggregate_uid,
                    collaborator_type, scanner_type, skip_ongoing_check):
        if not target_uri:
            raise APIError('not target_uri specified', status.HTTP_400_BAD_REQUEST)
        else:
            if (not target_uri.startswith('http://')) and (not target_uri.startswith('https://')):
                target_uri = 'https://'+target_uri
            url_validator = URLValidator()
            try:
                url_validator(target_uri)
            except ValidationError:
                raise APIError('invalid target_uri format: %s' % target_uri, status.HTTP_400_BAD_REQUEST)

        # abc_id is required parameter for non-crasher scans
        # TODO: fix Aqua first
#        if not is_prod and not abc_id:
#            raise APIError('Please, pass correct abc_id', status.HTTP_400_BAD_REQUEST)

        # Check if target is properly set.
        if not target_name and not target_uri:
            raise APIError('no target or uri specified', status.HTTP_400_BAD_REQUEST)
        else:
            # prevent adding unknown services as children to all the others
            if target_name in ['None', 'Unknown']:
                target_name = ''

            target = None
            if target_name:
                # first search by name
                try:
                    target = Target.objects.get(name__iexact=target_name)
                except Target.MultipleObjectsReturned:
                    target_qs = Target.objects.filter(name__iexact=target_name)
                    for t in target_qs:
                        if t.parent:
                            target = t.parent
                            break
                        else:
                            target = t
                            break
                except Target.DoesNotExist:
                    pass

            if not target:
                # try find target by slug
                target_qs = Target.objects.filter(slug=slugify_target_url(target_uri)).order_by('-id')[:1]
                if target_qs:
                    target = target_qs[0]

            if not target:
                # then try to find existing scan by URL slug and determine target from it
                scan_qs = Scan.objects.filter(slug=slugify_target_url(target_uri)).exclude(target__id=0).order_by('-id')[:2]
                for scan_item in scan_qs:
                    try:
                        target = scan_item.target
                    except Exception:
                        pass
                    else:
                        break

            # finally try by full url and create new if nothing found
            if not target:
                try:
                    target = Target.objects.get(url=target_uri)
                except Target.DoesNotExist:
                    if not target_name:
                        target_name = target_uri
                    # we create new target
                    target = Target(url=target_uri, slug=slugify_target_url(target_uri), name=target_name)
                    target.save()
                except Target.MultipleObjectsReturned:
                    target_qs = Target.objects.filter(name=target_uri)[:1]
                    if target_qs:
                        target = target_qs[0]

        if not target:
            raise APIError('no target specified', status.HTTP_400_BAD_REQUEST)

        if not target_name or target_name in ['None', 'Unknown']:
            target_name = target.name

        if target.parent:
            target = target.parent
            target_name = target.name

        if target.exclude:
            raise APIError('This target is excluded from the scan', status.HTTP_400_BAD_REQUEST)

        auth_profile = None
        if auth_profile_uid:
            try:
                auth_profile = AuthProfile.objects.get(uid=auth_profile_uid)
            except AuthProfile.DoesNotExist:
                pass

        scan_task = None
        try:
            if auth_profile:
                scan_task = ScanTask.objects.get(target=target, is_prod=is_prod, auth_profile=auth_profile)
            else:
                scan_task = ScanTask.objects.get(target=target, is_prod=is_prod)
        except ScanTask.DoesNotExist:
            pass
        except ScanTask.MultipleObjectsReturned:
            if auth_profile:
                scan_tasks = ScanTask.objects.filter(target=target, is_prod=is_prod, auth_profile=auth_profile)[:1]
            else:
                scan_tasks = ScanTask.objects.filter(target=target, is_prod=is_prod)[:1]
            if scan_tasks:
                scan_task = scan_tasks[0]
        if not scan_task:
            task_name = 'Scan %s with %s' % (target_name, ', '.join(profiles))
            scan_task = ScanTask(target=target, name=task_name, is_prod=is_prod, user=request.user)
            if auth_profile:
                scan_task.auth_profile = auth_profile
            scan_task.save()

        if not profiles:
            for p in scan_task.profiles.all():
                profiles.append(p.name)

        # use default scan profile if nothing specified by the user
        if not profiles:
            profiles.append('Yandex')

        for profile_name in profiles:
            try:
                scan_profile = ScanProfile.objects.get(name=profile_name)
            except ScanProfile.DoesNotExist:
                scan_task.delete()
                raise APIError('profile "%s" not found' % profile_name, status.HTTP_404_NOT_FOUND)
            else:
                ProfilesTasks.objects.get_or_create(scan_profile=scan_profile, scan_task=scan_task)

        # ABC ID specified
        if abc_id and not target.abc_id:
            try:
                target.abc_id = int(abc_id)
                target.save()
            except Exception:
                url = abc_id
                if '/' in abc_id:
                    abc_id = get_abc_id_by_url(url)
                else:
                    abc_id = get_abc_id_by_slug(url)
            if isinstance(abc_id, int) and abc_id != 0:
                target.abc_id = abc_id
                target.save()

        # ST queue specified
        if st_queue and not target.st_queue:
            if '/' in st_queue:
                st_queue = filter(lambda x: x, st_queue.split('/'))[-1]
                st_queue = st_queue.split('-')[0]
            if '-' in st_queue:
                st_queue = st_queue.split('-')[0]
            if st_queue:
                target.st_queue = st_queue.upper()
                target.save()

        # rate-limit new scans per user and uri
        latest_old_scan_time = datetime.datetime.now() + relativedelta(minutes=-settings.API_RATELIMIT_MINUTES)

        if is_prod and not skip_ongoing_check:
            ongoing_scans = Scan.objects.filter(url=target_uri, start__gte=latest_old_scan_time)[:1]
        else:
            ongoing_scans = Scan.objects.filter(url=target_uri, status=Scan.ST_INPROGRESS,
                                                start__gte=latest_old_scan_time)[:1]
        if ongoing_scans:
            return ongoing_scans[0], False

        # do not start multiple scans for the same production target if configured
        if is_prod and target.no_parallel_scans:
            ongoing_scans = Scan.objects.filter(target=target, status=Scan.ST_INPROGRESS,
                                                start__gte=latest_old_scan_time)[:1]
            if ongoing_scans:
                return ongoing_scans[0], False

        scan = scan_task.create_scan(user=request.user, url=target_uri, auth_profile=auth_profile,
                                     no_auto_tickets=no_auto_tickets, min_severity=severity, scanner_type=scanner_type)

        scan_modified = False
        # removed this logic to target settings only
        if usernames:
            # TODO: remove old logic if new works correctly
            '''
            if (request.user.is_superuser or not target.users.all()) or (request.user in target.users.all()):
                existing_users = [u.username for u in target.users.all()]
                for login in usernames.split(','):
                    login = login.strip().replace('@yandex-team.ru', '')
                    if not login:
                        continue
                    if login in existing_users:
                        continue
                    try:
                        user = User.objects.get(username=login)
                    except User.DoesNotExist:
                        user = User(username=login, email='{}@yandex-team.ru'.format(login),
                                    last_login=datetime.datetime.now())
                        user.save()
                    if user not in target.users.all():
                        target.users.add(user)
                    existing_users.append(login)
            '''
            existing_users = [u.username for u in target.users.all()]
            for login in usernames.split(','):
                login = login.strip().replace('@yandex-team.ru', '')
                if not login:
                    continue
                if login in existing_users:
                    continue
                try:
                    user = User.objects.get(username=login)
                except User.DoesNotExist:
                    user = User(username=login, email='{}@yandex-team.ru'.format(login),
                                last_login=datetime.datetime.now())
                    user.save()
                if user not in target.users.all() and not target.abc_id:
                    target.users.add(user)
                existing_users.append(login)

        if urls:
            qs = TargetUriMap.objects.filter(name=hashlib.sha256(json.dumps(urls).encode('utf-8')).hexdigest())[:1]
            if qs:
                tmap = qs[0]
            else:
                tmap = TargetUriMap(name=hashlib.sha256(json.dumps(urls).encode('utf-8')).hexdigest(), map='\n'.join(urls))
                tmap.save()

            sq = RequestSamples.objects.filter(format=RequestSamples.FMT_JSON,
                                               url=settings.TARGET_MAP_PREFIX + tmap.uid)[:1]
            if not sq:
                samples, new = RequestSamples.objects.get_or_create(format=RequestSamples.FMT_JSON,
                                                                    url=settings.TARGET_MAP_PREFIX + tmap.uid)
                if new:
                    samples.save()
            else:
                samples = sq[0]
            scan.sample_requests = samples
            scan_modified = True

        if st_ticket:
            scan.report_ticket = st_ticket
            scan_modified = True

        if collaborator_type:
            scan.collaborator_type = collaborator_type
            scan_modified = True

        if target_map_uid and not urls:
            qs = TargetUriMap.objects.filter(uid=target_map_uid)[:1]
            if qs:
                uri_map = qs[0]
                sample_qs = RequestSamples.objects.filter(format=RequestSamples.FMT_JSON,
                                                          url=settings.TARGET_MAP_PREFIX + uri_map.uid)[:1]
                if sample_qs:
                    scan.sample_requests = sample_qs[0]
                    scan_modified = True
                else:
                    samples = RequestSamples(format=RequestSamples.FMT_JSON,
                                             url=settings.TARGET_MAP_PREFIX + uri_map.uid)
                    samples.save()
                    scan.sample_requests = samples
                    scan_modified = True

        if request_samples_uid:
            try:
                samples = RequestSamples.objects.get(uid=request_samples_uid)
            except Exception:
                pass
            else:
                scan.sample_requests = samples
                scan_modified = True

        if request_samples_aggregate_uid:
            if not target.antirobot_uid:
                target.antirobot_uid = request_samples_aggregate_uid
                target.save()
            if not request_samples_resource or not request_samples_resource_type:
                res = get_latest_aggregate(target.antirobot_uid)
                request_samples_resource = 'sandbox-resource:{}'.format(res.get('id'))
                request_samples_resource_type = RequestSamples.FMT_JSON2

        if request_samples_resource and request_samples_resource_type:
            samples, new = RequestSamples.objects.get_or_create(format=request_samples_resource_type,
                                                                url=request_samples_resource)
            if new:
                samples.save()
            scan.sample_requests = samples
            scan_modified = True

        if ignore_time_limit:
            scan.ignore_time_limit = True
            scan_modified = True

        if qs_params:
            scan.qs_params = qs_params.split('\n')[0]
            scan_modified = True

        if user_agent:
            scan.user_agent = user_agent.split('\n')[0]
            scan_modified = True

        if send_mail:
            scan.send_mail = True
            scan_modified = True

        if rps > 0:
            scan.throttle = 60000 / rps
            scan_modified = True

        if scan_modified:
            scan.save()

        return scan, True

    @method_decorator(csrf_exempt)
    def dispatch(self, request, *args, **kwargs):
        return super(ScanHandler, self).dispatch(request, *args, **kwargs)


class ScanList(APIView):
    authentication_classes = (YandexTokenAuthentication,)
    permission_classes = (IsYandexUser,)

    def get_name(self):
        return 'Molly scan list api'

    def get(self, request, *args, **kwargs):
        res = []
        limit = request.GET.get('limit', 10)
        target_id = request.GET.get('target_id')
        if request.user.is_superuser:
            if target_id:
                qs = Scan.objects.filter(target__id=target_id).order_by('-start')[:limit]
            else:
                qs = Scan.objects.filter().order_by('-start')[:limit]
        else:
            if target_id:
                qs = Scan.objects.filter(user=request.user, target__id=target_id).order_by('-start')[:limit]
            else:
                qs = Scan.objects.filter(user=request.user).order_by('-start')[:limit]
        for s in qs:
            res.append({'id': s.uid, 'status': s.status})
        return Response({'scan_list': res})


class VulnTypeInfoHandler(APIView):
    def get_name(self):
        return 'Molly vuln info templates'

    def get(self, request, *args, **kwargs):
        vuln_type_id = request.GET.get('vuln_type_id')
        vuln_type = get_object_or_404(VulnerabilityType, id=vuln_type_id)
        res = dict()
        res['name'] = vuln_type.name
        res['summary'] = vuln_type.summary
        res['description'] = vuln_type.description
        res['manual_description'] = vuln_type.manual_description
        res['severity'] = vuln_type.severity
        res['description_url'] = vuln_type.description_url
        return Response(res)


class TicketsList(APIView):
    authentication_classes = (YandexTokenAuthentication,)
    permission_classes = (IsSuperUser,)

    def get_name(self):
        return 'Molly users'

    def get(self, request, *args, **kwargs):
        latest_ticket_time = datetime.datetime.now() + relativedelta(days=-7)
        tickets = VulnTicket.objects.filter(last_updated__gte=latest_ticket_time)
        res = []
        for t in tickets:
            res.append({'key': t.ticket_id, 'status': t.ticket_status,
                        'last_updated': t.last_updated, 'resolution': t.resolution})
        return Response(res)


class CrasherHandler(APIView):
    authentication_classes = (YandexTokenAuthentication,)
    permission_classes = (IsStaffUser,)
    parser_classes = (FormParser,)

    def get_name(self):
        return 'Crasher API'

    def get(self, request, *args, **kwargs):
        # XXX: 1?
        crasher = get_object_or_404(CrasherStatus, id=1)
        res = dict()
        res['status'] = crasher.status
        res['last_updated'] = crasher.last_updated.strftime('%Y-%m-%dT%H:%M:%S')
        res['user'] = crasher.user.username
        return Response(res)

    def put(self, request, *args, **kwargs):
        # XXX: 1?
        res = dict()
        crasher = get_object_or_404(CrasherStatus, id=1)
        status = request.data.get('status', 0)
        if status == CrasherStatus.STATUS_NEEDRUN:
            return Response(res)
        crasher.status = status
        if request.user.is_authenticated():
            crasher.user = request.user
        crasher.save()
        res['status'] = crasher.status
        res['last_updated'] = crasher.last_updated.strftime('%Y-%m-%dT%H:%M:%S')
        return Response(res)


class SampleUploadHandler(APIView):
    authentication_classes = (YandexTokenAuthentication,)
    permission_classes = (IsYandexUser,)
    parser_classes = (MultiPartParser, FormParser)

    def get_name(self):
        return 'Sample requests upload handler'

    def post(self, request, *args, **kwargs):
        res = dict()
        if 'sample_file' not in list(request.FILES.keys()):
            return Response({'error': 'Sample file not specified'}, status=status.HTTP_400_BAD_REQUEST)

        if 'sample_format' not in list(request.POST.keys()) or \
                        request.POST.get('sample_format') not in list(map(str, list(RequestSamples.SAMPLE_FORMATS.keys()))):
            return Response({'error': 'Sample format not specified or incorrect'}, status=status.HTTP_400_BAD_REQUEST)

        uploaded_file = request.FILES['sample_file']
        sample_format = request.POST.get('sample_format', RequestSamples.FMT_JSON)

        el_data = ''
        for chunk in uploaded_file.chunks():
            el_data += chunk

        ttl = 30
        elliptics_url = upload_to_elliptics(hashlib.sha256(el_data).hexdigest(), el_data, public=True, ttl=ttl)
        if not elliptics_url:
            return Response({'error': 'Elliptics error'}, status=status.HTTP_500_INTERNAL_SERVER_ERROR)

        sample, new = RequestSamples.objects.get_or_create(format=sample_format, url=elliptics_url)
        if new:
            sample.save()
        res['ok'] = True
        res['uid'] = sample.uid
        return Response(res)


class ScanQueueHandler(APIView):
    authentication_classes = (YandexTokenAuthentication,)
    permission_classes = (IsStaffUser,)

    def get_name(self):
        return 'Scan queue stat'

    def get(self, request, *args, **kwargs):
        res = dict()
        res['in_progress'] = Scan.objects.filter(status=Scan.ST_INPROGRESS).count()
        return Response(res)


class CrasherChunkHandler(APIView):
    authentication_classes = (YandexTokenAuthentication,)
    permission_classes = (IsStaffUser,)
    parser_classes = (MultiPartParser, FormParser)

    def get_name(self):
        return 'Crasher chink handler'

    def post(self, request, *args, **kwargs):
        res = dict()
        if 'data' not in list(request.FILES.keys()):
            return Response({'error': 'Chunk not specified'}, status=status.HTTP_400_BAD_REQUEST)

        uploaded_file = request.FILES['data']
        el_data = ''
        for chunk in uploaded_file.chunks():
            el_data += chunk

        obj = CrasherChunk(data=el_data)
        obj.save()

        res['ok'] = True
        res['uid'] = obj.uid
        return Response(res)

    def get(self, request, *args, **kwargs):
        if kwargs.get('uid'):
            chunk = get_object_or_404(CrasherChunk, uid=kwargs.get('uid'))
            return Response({'data': chunk.data})

        limit = 0
        try:
            limit = int(request.GET.get('limit', 3))
        except Exception:
            pass
        if limit > 0:
            qs = CrasherChunk.objects.filter().order_by('-pk')[:limit]
        else:
            qs = CrasherChunk.objects.filter().order_by('-pk')
        res = []
        for item in qs:
            res.append({'uid': item.uid})
        return Response(res)

    def delete(self, request, *args, **kwargs):
        if not kwargs.get('uid'):
            return Response({'error': 'uid not specified'}, status=status.HTTP_400_BAD_REQUEST)

        res = dict()
        res['ok'] = True
        try:
            CrasherChunk.objects.filter(uid=kwargs.get('uid')).delete()
        except Exception:
            res['ok'] = False
        return Response(res)


class EnvList(APIView):
    authentication_classes = (YandexTokenAuthentication,)
    permission_classes = (IsYandexUser,)

    def get_name(self):
        return 'List Qloud environment ids, targets and their urls with Molly enabled'

    def get(self, request, *args, **kwargs):
        res = []
        mc = memcache.Client([settings.MEMCACHE_SERVER], debug=0)
        qloud_cache = mc.get('molly_qloud_cache')
        if qloud_cache:
            try:
                res = json.loads(zlib.decompress(qloud_cache))
            except Exception:
                mc.delete('molly_qloud_cache')
        return Response({'env_list': res})


class StWebhookHandler(APIView):
    parser_classes = (JSONParser,)

    def get_name(self):
        return 'ST webhook endpoint'

    def post(self, request, *args, **kwargs):
        if not request.data.get('event'):
            return Response({'error': 'invalid request'}, status=status.HTTP_400_BAD_REQUEST)
        ticket_id = request.data.get('issue', {}).get('key')
        if not ticket_id:
            return Response({'error': 'invalid request'}, status=status.HTTP_400_BAD_REQUEST)
        LOG.debug("Tracker webhook: %s", str(ticket_id))
        try:
            ticket = VulnTicket.objects.get(ticket_id=ticket_id, tracker_type=VulnTicket.TT_ST)
        except Exception:
            catch_error()
            return Response({'error': 'stop retry'}, status=status.HTTP_400_BAD_REQUEST)
        st = Startrek(useragent=settings.ST_USER_AGENT, base_url=settings.ST_URL, token=settings.ST_OAUTH_TOKEN)
        try:
            res = st.issues[ticket_id]
        except Exception as e:
            catch_error()
            LOG.critical("Tracker error: %s", str(e))
            return Response({'error': 'invalid request'}, status=status.HTTP_400_BAD_REQUEST)
        if not res:
            return Response({'error': 'invalid request'}, status=status.HTTP_400_BAD_REQUEST)
        ticket_status = ticket.get_status(res.status.key)
        if ticket.ticket_status == VulnTicket.ST_CLOSED and ticket_status == VulnTicket.ST_OPEN:
            ticket.resolution = 'reopened'
        elif res.resolution:
            ticket.resolution = res.resolution.key.lower()
            ticket.ticket_status = ticket_status
            ticket.save()
        return Response({'status': 'ok'})


class TargetMapExportView(APIView):
    def get(self, request, *args, **kwargs):
        res = []
        target_map = get_object_or_404(TargetUriMap, uid=kwargs.get('uid'))
        for item in target_map.map.split('\n'):
            item = item.strip()
            if not item:
                continue
            try:
                urlinfo = urllib.parse.urlparse(item)
                if urlinfo:
                    rec = dict()
                    rec["method"] = "GET"
                    rec["body"] = ""
                    rec["path"] = urlinfo.path
                    rec["rawquery"] = urlinfo.path
                    rec["proto"] = "HTTP/1.0"
                    rec["headers"] = []
                    res.append(rec)
            except Exception:
                raise
        return Response(res)

