# -*- coding: utf-8 -*-
import memcache
import json
import datetime
import zlib

from django.conf import settings
from django import forms
from django.core.exceptions import ValidationError
from django.db.models import Q
from django.contrib.auth.models import User
from django.shortcuts import get_object_or_404, redirect
from django.utils.translation import ugettext_lazy as _

from .models import ScanTask, Scan, Profile, ProfilesTargets, Vulnerability, VulnerabilityType, \
    Target, VulnTicket, AuthProfile, TargetUriMap, ScanProfile
from .utils import get_abc_scopes


class AuthProfileForm(forms.ModelForm):
    class Meta:
        model = AuthProfile
        exclude = ('uid', 'users')

    def __init__(self, *args, **kwargs):
        super(AuthProfileForm, self).__init__(*args, **kwargs)
        self.fields['password'].initial = ''
        self.fields['password'].help_text = 'Put password or OAuth token here.'

    def clean_passport_host(self):
        passport_host = self.cleaned_data.get('passport_host', '').strip()
        if not passport_host:
            raise ValidationError('Passport host not set ')
        #XXX: regexp?
        if ('yandex.' not in passport_host) and ('yandex-team.ru' not in passport_host) and \
                ('pricelabs.ru' not in passport_host) and ('beru.ru' not in passport_host):
            raise ValidationError('Invalid Passport port')
        return passport_host


class TargetUriMapForm(forms.ModelForm):
    class Meta:
        model = TargetUriMap
        exclude = ('uid', 'users')

    def __init__(self, *args, **kwargs):
        super(TargetUriMapForm, self).__init__(*args, **kwargs)


class NewScanForm(forms.Form):
    """
    UI init new scan form
    """
    def __init__(self, user, *args, **kwargs):
        super(NewScanForm, self).__init__(*args, **kwargs)
        self.user = user
        known = []
        self.fields['auth_profile_uid'].choices = [(0, _('Unauthenticated'))]
        self.user_abc_ids = []
        if self.user.is_superuser:
            self.fields['report_ticket'] = forms.CharField(max_length=64, required=False, label=_("Report to ST"))
            self.fields['scanner_type'] = forms.ChoiceField(choices=[(1, 'Burp'), (2, 'Burp 2')], label=_("Scanner"))
            self.fields['collaborator_type'] = forms.ChoiceField(choices=list(Scan.COLLAB_TYPES.items()),
                                                                 required=False, label=_("Collaborator"))
            user_auth_profiles = AuthProfile.objects.all()
        else:
            default_profile = AuthProfile.objects.get(uid=settings.DEFAULT_AUTH_PROFILE)
            self.fields['auth_profile_uid'].choices.append((default_profile.uid,
                                                            '{} ({})'.format(default_profile.name,
                                                                              default_profile.username)))
            known.append(settings.DEFAULT_AUTH_PROFILE)
            user_auth_profiles = AuthProfile.objects.filter(users__in=[user.id])
            self.user_abc_ids = [item[0] for item in get_abc_scopes(self.user.username)]
        for item in user_auth_profiles:
            if item.uid in known:
                continue
            self.fields['auth_profile_uid'].choices.append((item.uid, '{} ({})'.format(item.name, item.username)))
            known.append(item.uid)
        target_choices = []
        qs = Target.objects.filter(parent__isnull=True).order_by('-name')
        for item in qs:
            target_choices.append((item.id, item.name))
#        self.fields['target_id'].choices = target_choices
        scan_profile_choices = [(p.id, p.name) for p in ScanProfile.objects.filter(is_public=True)]
        self.fields['scan_profile_id'].choices = scan_profile_choices
        self.fields['abc_id'].choices = []
        try:
            mc = memcache.Client([settings.MEMCACHE_SERVER], debug=0)
            abc = mc.get('molly_abc_cache')
            if abc:
                services = json.loads(zlib.decompress(abc))
                for item in services:
                    if not item.get('id', 0) or not item.get('name', {}).get('ru') or item.get('is_deleted'):
                        continue
                    if self.user_abc_ids and item.get('id') not in self.user_abc_ids:
                        continue
                    self.fields['abc_id'].choices.append((item.get('id'),
                                                          '{}. {}'.format(item.get('id'),
                                                                           item.get('name', {}).get('ru'))))
        except Exception:
            pass

    abc_id = forms.ChoiceField(choices=(), required=True, label=_("ABC"))
    url = forms.URLField(required=True, label=_('URL'))
    scan_profile_id = forms.ChoiceField(choices=(), required=True, label=_("Scanner profile"))
    auth_profile_uid = forms.ChoiceField(choices=(), required=True)
    rps = forms.IntegerField(required=False, initial=0, label=_("Max RPS"), help_text=_("0 - unlimited"))


class TargetMergeForm(forms.Form):
    def __init__(self, *args, **kwargs):
        super(TargetMergeForm, self).__init__(*args, **kwargs)
        target_ids = []
        qs = Target.objects.filter(parent__isnull=True).order_by('-name')
        for item in qs:
            target_ids.append((item.id, '{} ({})'.format(item.name, item.children.count())))
        self.fields['target_id'].choices = target_ids
        self.fields['merge_id'].choices = target_ids

    target_id = forms.ChoiceField(choices=(), required=True, label=_("Parent"))
    merge_id = forms.ChoiceField(choices=(), required=True, label=_("Child"))


class AntirobotWizardForm(forms.Form):
    def __init__(self, *args, **kwargs):
        super(AntirobotWizardForm, self).__init__(*args, **kwargs)
        target_choices = []
        qs = Target.objects.filter(parent__isnull=True).order_by('name')
        for item in qs:
            if not item.name:
                continue
            if item.name.startswith('http://') or item.name.startswith('https://'):
                continue
            if not item.abc_id:
                continue
            target_choices.append((item.id, item.name))
        self.fields['target_id'].choices = target_choices

    target_id = forms.ChoiceField(choices=(), required=True, label=_("Service"))


class CrasherStatusForm(forms.Form):
    run = forms.BooleanField(label=_("Run crasher"), required=False)


class TargetEditForm(forms.Form):
    name = forms.CharField(max_length=128, required=True, label=_("Name"))
    st_queue = forms.CharField(max_length=128, required=False, label=_("ST queue"))
    abc_id = forms.ChoiceField(choices=(), required=True, label=_("ABC"))
    contacts = forms.CharField(max_length=255, required=False, label=_("Contacts"),
                               help_text=_('This people will receive scan notifications.'))
    rps = forms.IntegerField(required=False, initial=0, label=_("Max RPS"), help_text=_("0 - unlimited"))
    notify_on_scan = forms.BooleanField(label=_("Send mail notification on every scan start"), required=False)
    no_parallel_scans = forms.BooleanField(label=_("Do not scan on multiple domains"), required=False)
    log_aggregation_uid = forms.CharField(max_length=128, required=False, label=_("Log aggregation uid"))

    def __init__(self, target, user, *args, **kwargs):
        super(TargetEditForm, self).__init__(*args, **kwargs)
        self.target = target
        self.user = user
        self.user_abc_ids = []
        self.fields['name'].initial = self.target.name
        self.fields['st_queue'].initial = self.target.st_queue
        self.fields['rps'].initial = self.target.rps_limit
        self.fields['notify_on_scan'].initial = self.target.notify_on_scan
        self.fields['log_aggregation_uid'].initial = self.target.antirobot_uid
        if self.user.is_superuser:
            self.fields['exclude'] = forms.BooleanField(label=_("Exclude from Crasher"), required=False)
            self.fields['exclude'].initial = self.target.exclude
            self.fields['url'] = forms.CharField(max_length=128, required=False, label=_("URL"))
            self.fields['url'].initial = self.target.url
            self.fields['slug'] = forms.CharField(max_length=128, required=False, label=_("Slug"))
            self.fields['slug'].initial = self.target.slug
        else:
            self.user_abc_ids = [item[0] for item in get_abc_scopes(self.user.username)]
        contacts = set([str(u.username) for u in self.target.users.all()])
#        if self.target.children.count():
#            for child in self.target.children.all():
#                for u in child.users.all():
#                    contacts.add(str(u.username))
        self.fields['contacts'].initial = ', '.join(list(contacts))
        self.fields['abc_id'].choices = []
#        (0, u'Unknown')]
        self.fields['no_parallel_scans'].initial = self.target.no_parallel_scans
        known = []
        try:
            mc = memcache.Client([settings.MEMCACHE_SERVER], debug=0)
            abc = mc.get('molly_abc_cache')
            if abc:
                services = json.loads(zlib.decompress(abc))
                for item in services:
                    if not item.get('id', 0) or not item.get('name', {}).get('ru') or item.get('is_deleted'):
                        continue
                    if self.user_abc_ids and item.get('id') not in self.user_abc_ids:
                        continue
                    self.fields['abc_id'].choices.append((item.get('id'),
                                                          '{}. {}'.format(item.get('id'),
                                                                           item.get('name', {}).get('ru'))))
                    known.append(item.get('id'))
        except Exception:
            pass
        if self.target.abc_id and self.target.abc_id in known:
            self.fields['abc_id'].initial = self.target.abc_id

    def save(self):
        self.target.name = self.cleaned_data.get('name')
        self.target.st_queue = self.cleaned_data.get('st_queue')
        if self.cleaned_data.get('rps', 0):
            self.target.max_rps = int(self.cleaned_data.get('rps'))
        if self.user.is_superuser:
            self.target.exclude = self.cleaned_data.get('exclude', False)
        if self.cleaned_data.get('abc_id', 0):
            self.target.abc_id = int(self.cleaned_data.get('abc_id'))
        if self.cleaned_data.get('log_aggregation_uid', ''):
            self.target.antirobot_uid = self.cleaned_data.get('log_aggregation_uid', '')
        self.target.notify_on_scan = self.cleaned_data.get('notify_on_scan', False)
        self.target.no_parallel_scans = self.cleaned_data.get('no_parallel_scans', False)
        self.target.save()

        existing_users = []
        usernames = self.cleaned_data.get('contacts', '').lower()
        self.target.users.clear()
        for login in usernames.split(','):
            login = login.strip().replace('@yandex-team.ru', '')
            if not login:
                continue
            if login in existing_users:
                continue
            try:
                user = User.objects.get(username=login)
            except User.DoesNotExist:
                user = User(username=str(login), email=login + '@yandex-team.ru',
                            last_login=datetime.datetime.now())
                user.save()
            if user not in self.target.users.all():
                self.target.users.add(user)
            for child in self.target.children.all():
                child.users.clear()
                child.users.add(user)
            existing_users.append(login)


class ScanTargetEditForm(forms.Form):
    target_id = forms.ChoiceField(choices=(), required=True, label=_("Target"))

    def __init__(self, scan, user, *args, **kwargs):
        super(ScanTargetEditForm, self).__init__(*args, **kwargs)
        self.scan = scan
        self.user = user
        target_ids = []
        qs = Target.objects.filter(parent__isnull=True).order_by('-name')
        for item in qs:
            if not self.user.is_superuser:
                #XXX: do not allow crasher autogen as targets
                if 'CRASHER_' in item.name:
                    continue
            target_ids.append((item.id, '{} {} {}'.format(item.name,
                                                           '(%s)' % item.st_queue if item.st_queue else '',
                                                           '(%d)' % item.abc_id if item.abc_id else '')))
        self.fields['target_id'].choices = target_ids
        self.fields['target_id'].initial = self.scan.target.id

    def save(self):
        target = get_object_or_404(Target, id=self.cleaned_data.get('target_id'))
        qs = Scan.objects.filter(start__gte=self.scan.start, target=self.scan.target)
        for item in qs:
            if item.slug == self.scan.slug:
                item.target = target
                item.save()
        self.scan.target = target
        self.scan.save()


class NewTargetForm(forms.Form):
    name = forms.CharField(max_length=128, required=True, label=_("Name"))
    st_queue = forms.CharField(max_length=128, required=False, label=_("ST queue"))
    abc_id = forms.ChoiceField(choices=(), required=True, label=_("ABC"))

    def __init__(self, user, *args, **kwargs):
        super(NewTargetForm, self).__init__(*args, **kwargs)
        self.user = user
        self.fields['abc_id'].choices = []
        self.user_abc_ids = []
        if not self.user.is_superuser:
            self.user_abc_ids = [item[0] for item in get_abc_scopes(self.user.username)]
        known = []
        try:
            mc = memcache.Client([settings.MEMCACHE_SERVER], debug=0)
            abc = mc.get('molly_abc_cache')
            if abc:
                services = json.loads(zlib.decompress(abc))
                for item in services:
                    if not item.get('id', 0) or not item.get('name', {}).get('ru') or item.get('is_deleted'):
                        continue
                    if self.user_abc_ids and item.get('id') not in self.user_abc_ids:
                        continue
                    self.fields['abc_id'].choices.append((item.get('id'),
                                                          '{}. {}'.format(item.get('id'),
                                                                           item.get('name', {}).get('ru'))))
                    known.append(item.get('id'))
        except Exception:
            pass

    def save(self):
        return
