

import re
import uuid
import json
import traceback

from datetime import datetime
from dateutil.relativedelta import relativedelta

from logging import getLogger
from tempfile import NamedTemporaryFile

from urllib.parse import urlparse, urlunparse
from django.db import models
from django.contrib.auth.models import User, Group
from django.utils.translation import ugettext_lazy as _
from django.conf import settings
from django.template import Template, Context
from django.utils import timezone

from startrek_client import Startrek

from .notification.errors import catch_error
from .utils import slugify_target_url, pg_unwrap, log_scan_to_splunk

logger = getLogger(__name__)


class Profile(models.Model):
    user = models.OneToOneField(settings.AUTH_USER_MODEL, unique=True, related_name='profile', on_delete=models.CASCADE)
    list_per_page = models.PositiveIntegerField(null=False, default=50)
    lang_ui = models.CharField(_('Interface language'), max_length=4, choices=settings.LANGUAGES,
                               default=settings.DEFAULT_LANGUAGE)

    class Meta:
        db_table = 'w3af_webui_profile'


def create_profile(sender, **kwargs):
    if kwargs.get('created'):
        user = kwargs.get('instance')   # type: User
        if not user.email:
            user.email = str(user.username) + '@yandex-team.ru'
            user.save()

        Profile.objects.create(user=user)
        group = Group.objects.get(name='User')
        user.groups.add(group)


def remove_profile(sender, **kwargs):
    user = kwargs.get('instance')   # type: User
    if not user:
        return
    try:
        profile = Profile.objects.get(user__id=user.id)
        profile.delete()
    except Exception:
        pass


def fix_login_time(sender, **kwargs):
    user = kwargs.get('instance')   # type: User
    if not user:
        return
    user.last_login = timezone.now()


models.signals.pre_save.connect(fix_login_time, sender=User)
models.signals.post_save.connect(create_profile, sender=User)
models.signals.pre_delete.connect(remove_profile, sender=User)


class ScanProfile(models.Model):
    """
        Scan profiles here
    """
    name = models.CharField(_('Name'), unique=True, max_length=240)
    short_comment = models.CharField(_('Short description'), blank=True, max_length=240)
    burp_profile = models.TextField(_('Burp-profile'), blank=True, default="{}")
    user = models.ForeignKey(settings.AUTH_USER_MODEL, verbose_name=_('User'), blank=True, null=True, on_delete=models.CASCADE)
    is_public = models.BooleanField('Public', default=False)

    class Meta:
        verbose_name = _('scan profile')
        verbose_name_plural = _('Scan profiles')
        db_table = 'scan_profiles'

    def __str__(self):
        return '%s' % self.name


class Target(models.Model):
    """
        Scan target
    """
    name = models.CharField(_('Name'), unique=True, max_length=255)
    parent = models.ForeignKey('self', blank=True, null=True, related_name="children", on_delete=models.CASCADE)
    url = models.CharField(_('URL'), max_length=1024, help_text= '<p class=checkRes></p>')
    comment = models.CharField(_('Comment'), max_length=255, blank=True, null=True)
    users = models.ManyToManyField(settings.AUTH_USER_MODEL, blank=True)
    last_scan = models.DateTimeField(_('Last scan date'), blank=True, null=True, editable=False)
    st_queue = models.CharField(_('ST QUEUE'), blank=True, null=True, max_length=255)
    abc_id = models.PositiveIntegerField(_('ABC ID'), default=0, blank=True, null=True)
    antirobot_uid = models.CharField(_('Antirobot header value'), max_length=255, null=True, blank=True)
    slug = models.CharField(_('URL Slug'), max_length=1024, default="/")
    max_rps = models.PositiveIntegerField(_('RPS limit'), default=0, blank=True, null=True)
    exclude = models.BooleanField('Do not scan', default=False)
    notify_on_scan = models.BooleanField('Notify on scan start', default=False)
    last_email = models.DateTimeField(_('Last email sent'), blank=True, null=True, editable=False)
    no_parallel_scans = models.BooleanField('Scan only first domain', default=False)

    class Meta:
        ordering = ['name']
        verbose_name = _('target')
        verbose_name_plural = _('Targets')
        db_table = 'targets'

    def __str__(self):
        return '%s' % self.name

    @property
    def rps_limit(self):
        try:
            int(self.max_rps)
        except Exception:
            return 0
        return int(self.max_rps)

    @property
    def is_prod(self):
        # if target was not manually scanned for 1 week - it's not really connected to Molly
        deadline = datetime.now() + relativedelta(days=-7)
        target = self
        if self.parent:
            target = self.parent
        return Scan.objects.filter(is_prod=False, target=target,
                                   start__gte=deadline).exists()

    @property
    def last_scan_obj(self):
        target = self
        if self.parent:
            target = self.parent
        qs = Scan.objects.filter(target=target).order_by('-finish')[:1]
        if not qs:
            return None
        return qs[0]

    @property
    def last_scan_vulns(self):
        scan = self.last_scan_obj
        if not scan:
            return -1
        return len(scan.get_vulnerabilities(new_only=False, ignore_fp=True))


class ProfilesTargets(models.Model):
    scan_profile = models.ForeignKey(ScanProfile, blank=False, verbose_name=_('Profile'), on_delete=models.CASCADE)
    target = models.ForeignKey(Target, blank=True, verbose_name=_('Target'), on_delete=models.CASCADE)

    class Meta:
        verbose_name = _('Default scan profile')
        verbose_name_plural = _('Default scan profiles')
        db_table = 'profiles_targets'

    def __str__(self):
        return '%s' % self.scan_profile


class AuthProfile(models.Model):
    AUTH_PASSWORD = 0
    AUTH_OAUTH = 1
    AUTH_TYPES = {
        AUTH_PASSWORD: 'Password',
        AUTH_OAUTH: 'OAuth',
    }
    uid = models.CharField(_('Uid'), max_length=128, blank=False)
    name = models.CharField(_('Name'), max_length=128, blank=False)
    username = models.CharField(_('Username'), max_length=128, null=True)
    password = models.CharField(_('Password'), max_length=128, null=True)
    passport_host = models.CharField(_('Passport Host'), max_length=128, null=True)
    comment = models.TextField(_('Description'), blank=True, null=True, max_length=500)
    users = models.ManyToManyField(settings.AUTH_USER_MODEL, blank=True)
    auth_type = models.IntegerField(choices=list(AUTH_TYPES.items()), default=AUTH_PASSWORD)

    class Meta:
        verbose_name = _('Authentication Profile')
        verbose_name_plural = _('Authentication Profiles')
        db_table = 'w3af_webui_authprofile'

    def get_yandex_burp_profile(self, target_url=''):
        config = dict()
        config["auth_provider"] = "YANDEX"
        if self.auth_type == self.AUTH_OAUTH:
            config["auth_schema"] = "OAUTH",
            config["auth_host"] = ""
            config["auth_username"] = ""
            # sanitize it a bit
            config["auth_password"] = self.oauth_header
        elif self.auth_type == self.AUTH_PASSWORD:
            config["auth_schema"] = "PASSPORT"
            config["auth_host"] = self.get_normalized_passport_host(target_url)
            config["auth_username"] = self.normalized_username
            config["auth_password"] = self.normalized_password
        return config

    def get_normalized_passport_host(self, target_url=''):
        COMPLEX_TLDS = ['tr', 'ge', 'de', 'au']
        COOKIELESS_TLDS = ['net', 'yandex']
        if ((not self.passport_host.startswith('http://')) and (not self.passport_host.startswith('https://')) and (
                not self.passport_host.startswith('//'))):
            self.passport_host = 'https://' + self.passport_host
        p = urlparse(self.passport_host)
        passport_domain = p.netloc
        if ':' in p.netloc:
            passport_domain, port = p.netloc.split(':')
        ds = passport_domain.split('.')
        if ds[-1] in COMPLEX_TLDS:
            passport_prefix = '.'.join(ds[:-2])
        else:
            passport_prefix = '.'.join(ds[:-1])
        if target_url:
            p = urlparse(target_url)
            target_domain = p.netloc
            if ':' in p.netloc:
                target_domain, port = p.netloc.split(':')
            ds = target_domain.split('.')
            tld = ds[-1]
            if tld in COMPLEX_TLDS:
                tld = 'com.' + tld
            if tld in COOKIELESS_TLDS:
                return ''
            if tld not in settings.PASSPORT_TLDS:
                return ''
            passport_host = '.'.join([passport_prefix, tld])
        else:
            passport_host = passport_domain
        return passport_host

    @property
    def normalized_username(self):
        username = self.username
        username = username.replace('\n', '').strip()
        username = username.replace('@yandex.ru', '')
        username = username.replace('@yandex-team.ru', '')
        username = username.replace('@ya.ru', '')
        return username

    @property
    def normalized_password(self):
        if self.auth_type == AuthProfile.AUTH_OAUTH:
            return self.password.replace('\n', '').strip().split(' ')[-1]
        # XXX: ugly!
        return self.password.replace('\n', '').strip()

    @property
    def oauth_header(self):
        return 'Authorization: OAuth %s' % self.normalized_password

    def __str__(self):
        return '%s (%s)' % (self.name, self.uid)


class ScanTask(models.Model):
    """
        Scan jobs
    """
    name = models.CharField(_('Task name'), max_length=1024, null=True)
    user = models.ForeignKey(settings.AUTH_USER_MODEL, verbose_name=_('User'), blank=True, null=True, on_delete=models.CASCADE)
    target = models.ForeignKey(Target, verbose_name=_('Target'), related_name='scan_tasks', on_delete=models.CASCADE)
    comment = models.TextField(_('Description'), blank=True, null=True, max_length=500)
    profiles = models.ManyToManyField(ScanProfile, through='ProfilesTasks')
    is_prod = models.BooleanField('Prod', default=False)
    auth_profile = models.ForeignKey(AuthProfile, verbose_name=_('AuthProfile'), null=True, blank=True, on_delete=models.CASCADE)

    class Meta:
        verbose_name = _('scan task')
        verbose_name_plural = _('scan tasks')
        db_table = 'scan_tasks'

    def __str__(self):
        return '%s' % (self.name)

    def create_scan(self, user=None, url='', auth_profile=None, send_mail=False,
                    no_auto_tickets=False, min_severity=30, scanner_type=1):
        if not url:
            return None
        scan = Scan(scan_task=self, user=user, url=url, target=self.target, is_prod=self.is_prod,
                    send_mail=send_mail, no_auto_tickets=no_auto_tickets, min_ticket_severity=min_severity, scanner_type=scanner_type)
        if auth_profile:
            scan.auth_profile = auth_profile
        if self.target.rps_limit != 0:
            rps_limit = self.target.rps_limit
            # If someone set less then 5 rps (1 for example) - ignore it
            if self.target.rps_limit < 5:
                rps_limit = 5
            scan.throttle = 60000 / rps_limit
        scan.save()
        return scan


class ProfilesTasks(models.Model):
    scan_profile = models.ForeignKey(ScanProfile, verbose_name=_('Profile'), on_delete=models.CASCADE)
    scan_task = models.ForeignKey(ScanTask, verbose_name=_('Scan'), on_delete=models.CASCADE)

    class Meta:
        verbose_name = _('Profile')
        verbose_name_plural = _('Profiles')
        db_table = 'profiles_tasks'

    def __str__(self):
        return '%s' % (self.scan_profile, )


class VulnerabilityType(models.Model):
    SEVERITY_IGNORE = -1
    SEVERITY_UNKNOWN = 0
    SEVERITY_INFORMATION = 10
    SEVERITY_LOW = 20
    SEVERITY_MEDIUM = 30
    SEVERITY_HIGH = 40
    SEVERITY_CRITICAL = 50
    SEVERITY_BLOCKER = 60
    SEVERITIES = {
        SEVERITY_IGNORE: 'Not informative',
        SEVERITY_UNKNOWN: 'Unknown',
        SEVERITY_INFORMATION: 'Information',
        SEVERITY_LOW: 'Low',
        SEVERITY_MEDIUM: 'Medium',
        SEVERITY_HIGH: 'High',
        SEVERITY_CRITICAL: 'Critical',
        SEVERITY_BLOCKER: 'Blocker',
    }
    name = models.CharField(_('Type of vulnerability'), max_length=80)
    summary = models.TextField(_('Summary'), blank=True, null=True)
    short_summary = models.CharField(_('Short summary'), max_length=128)
    description = models.TextField(_('Description'), blank=True, null=True)
    description_url = models.CharField(_('URL with description'), max_length=1024, blank=True, null=True)
    manual_description = models.TextField(_('Description for manual vuln'), blank=True, null=True)
    severity = models.IntegerField(choices=list(SEVERITIES.items()), default=SEVERITY_UNKNOWN)
    combine = models.BooleanField('Combine vulnerabilities', default=False)
    archived = models.BooleanField('Archived', default=False)
    scanner_severity = models.CharField(_('Scanner severity'), max_length=64, null=True)
    is_internal = models.BooleanField('Internal vuln type', default=False)
    tracker_severity = models.IntegerField(default=SEVERITY_UNKNOWN)
    st_tags = models.CharField(_('Tracker tags'), max_length=128, null=True)

    class Meta:
        verbose_name = _('Type of vulnerability')
        verbose_name_plural = _('Types of vulnerability')
        db_table = 'vulnerability_types'

    def __str__(self):
        return '%s' % self.name

    @property
    def human_name(self):
        if self.short_summary:
            return self.short_summary
        return self.name

    @property
    def wiki_description(self):
        if not self.description:
            return ''
        description = re.sub(r'<b>([\s+]?)', '**', self.description)
        description = re.sub(r'([\s+]?)</b>', '**', description)
        description = description.replace('<br>', '\n')
        description = description.replace('<li>', '* ')
        description = description.replace('</li>', '')
        description = description.replace('<ul>', '\n')
        description = description.replace('</ul>', '\n')
        description = description.replace('<pre>', '%%')
        description = description.replace('</pre>', '%%')
        return description.strip()


class FalsePositiveByVulnType(models.Model):
    vuln_type = models.ForeignKey(VulnerabilityType, verbose_name=_('Vulnerability type'), null=True, on_delete=models.CASCADE)
    target = models.ForeignKey(Target, verbose_name=_('Target'), on_delete=models.CASCADE)
    deleted = models.BooleanField(default=False)
    last_updated = models.DateTimeField(null=True, auto_now=True)
    user = models.ForeignKey(settings.AUTH_USER_MODEL, verbose_name=_('User'), blank=True, null=True, on_delete=models.CASCADE)

    class Meta:
        db_table = 'w3af_webui_falsepositivebyvulntype'


class HTTPHeader(models.Model):
    name = models.CharField(_('Header name'), max_length=1024)
    value = models.CharField(_('Header value'), max_length=1024, blank=True)

    class Meta:
        verbose_name = _('HTTP header')
        verbose_name_plural = _('HTTP headers')
        db_table = 'w3af_webui_httpheader'

    def __str__(self):
        return '{}: {}'.format(self.name, self.safe_value)

    @property
    def safe_value(self):
        if self.name.lower() not in ('cookie', 'authorization'):
            return self.value
        OAUTH_RE = re.compile('^(OAuth|Bearer)\s')
        if OAUTH_RE.match(self.value):
            return self.value[:-8] + '.XXX'
        COOKIE_RE = re.compile('^(Session_id|sessionid2)\=')
        res = []
        for cookie_val in self.value.split(';'):
            cookie_val = cookie_val.strip()
            if COOKIE_RE.match(cookie_val):
                res.append('.'.join(cookie_val.split('.')[:-1]) + '.XXX')
                continue
            res.append(cookie_val)
        return '; '.join(res)


class HTTPTransaction(models.Model):
    request_line = models.CharField(_('Request line'), max_length=2048)
    request_headers = models.ManyToManyField(HTTPHeader, blank=True, related_name='requests')
    request_body = models.BinaryField(_('Request Body'), blank=True, null=True)
    status_line = models.CharField(_('Status line'), max_length=2048, null=True)
    response_headers = models.ManyToManyField(HTTPHeader, blank=True, related_name='responses')
    response_body = models.BinaryField(_('Response Body'), blank=True, null=True)

    class Meta:
        verbose_name = _('HTTP Transaction')
        verbose_name_plural = _('HTTP Transactions')
        db_table = 'w3af_webui_httptransaction'

    @property
    def raw_request_body(self):
        return bytes(self.request_body).decode("latin-1")

    @property
    def raw_response_body(self):
        return bytes(self.response_body).decode("latin-1")

    @property
    def uri(self):
        # we assume it's HTTP/1.1
        l = self.request_line.split(' ')
        if len(l) < 2:
            return ''
        return l[1]

    def __str__(self):
        return '%s' % self.request_line


class TriagedTicketsManager(models.Manager):
    def get_query_set(self):
        return super(TriagedTicketsManager, self).get_query_set().filter(
            ticket_status=VulnTicket.ST_CLOSED,
            resolution__in=VulnTicket.TRIAGING_RESOLUTIONS
        ).exclude(webhook_url='')


class RequestSamples(models.Model):
    """
        Object describing sample requests to be issued while scan
    """
    FMT_JSON = 0
    FMT_BURP = 1
    FMT_SERP = 2
    FMT_SERP2 = 3
    FMT_JSON2 = 4
    SAMPLE_FORMATS = {
        FMT_JSON: 'Molly-compatible format',
        FMT_BURP: 'Burp XML',
        FMT_SERP: 'SERP Sandbox',
        FMT_SERP2: 'SERP Sandbox tmp',
        FMT_JSON2: 'Molly-preconverted JSON format'
    }

    uid = models.CharField(_('Uid'), max_length=128, blank=False)
    url = models.CharField(_('Full scan log URL or resource'), max_length=1024, null=True, blank=True)
    format = models.IntegerField(choices=list(SAMPLE_FORMATS.items()), default=FMT_JSON)

    class Meta:
        verbose_name = _('Request Samples')
        verbose_name_plural = _('Request Samples')
        db_table = 'w3af_webui_requestsamples'

    @property
    def user_friendly_url(self):
        if self.url.startswith('sandbox-resource:'):
            splt_url = self.url.split(':')
            return 'https://sandbox.yandex-team.ru/resource/{id}/view'.format(id=splt_url[-1])
        return self.url

    def __str__(self):
        return '%s' % self.uid


def set_sample_meta(sender, **kwargs):
    samples = kwargs.get('instance')  # type: RequestSamples
    if kwargs.get('created'):
        if re.match('^[\d]+$', samples.url):
            samples.url = 'sandbox-resource:{}'.format(samples.url)
        samples.uid = str(uuid.uuid4())
        samples.save()


models.signals.post_save.connect(set_sample_meta, sender=RequestSamples)


class Scan(models.Model):
    """
        Reports for scans
    """
    SCANNER_W3AF = 0
    SCANNER_BURP = 1
    SCANNER_BURP2 = 2
    SCANNERS = {
        SCANNER_BURP: 'Burp',
        SCANNER_BURP2: 'Burp2'
    }

    ST_INPROGRESS = 1
    ST_DONE = 2
    ST_FAIL = 3
    ST_ABORTED = 4
    SCAN_STATUSES = {
        ST_INPROGRESS: _('In progress'),
        ST_DONE: _('Done'),
        ST_FAIL: _('Failed'),
        ST_ABORTED: _('Aborted')
    }

    COLLAB_TYPE_PRIVATE = 0
    COLLAB_TYPE_PUBLIC = 1
    COLLAB_TYPES = {
        COLLAB_TYPE_PUBLIC: _('Public'),
        COLLAB_TYPE_PRIVATE: _('Custom')
    }

    scan_task = models.ForeignKey(ScanTask, verbose_name=_('Scan task'), related_name="scans", on_delete=models.CASCADE)
    status = models.IntegerField(_('Status'), editable=True, default=ST_INPROGRESS, choices=list(SCAN_STATUSES.items()))
    start = models.DateTimeField(_('Scan start'), auto_now_add=True)
    finish = models.DateTimeField(_('Scan finish'), blank=True, null=True,)
    data = models.CharField(max_length=255, default='', verbose_name = _('Report'), null=True, blank=True)
    pidfile = models.CharField(_('Scanner process pidfile'), blank=True, null=True, max_length=255)
    last_updated = models.DateTimeField(null=True, auto_now=True)
    result_message = models.CharField(max_length=1000, null=True, default='', blank=True)
    user = models.ForeignKey(settings.AUTH_USER_MODEL, verbose_name=_('User'), blank=True, null=True, on_delete=models.CASCADE)
    show_report_time = models.DateTimeField(_('Show report time'), blank=True, null=True)
    uid = models.CharField(_('Scan uid'), blank=True, null=True, max_length=64)
    task_id = models.CharField(_('Task id in queue'), blank=True, null=True, max_length=64)
    url = models.CharField(_('URL'), max_length=1024)
    sample_requests = models.ForeignKey(RequestSamples, null=True, blank=True, on_delete=models.CASCADE)
    scan_log = models.CharField(_('Full scan log URL'), max_length=1024, null=True, blank=True)
    auth_profile = models.ForeignKey(AuthProfile, verbose_name=_('AuthProfile'), null=True, blank=True, on_delete=models.CASCADE)
    ignore_time_limit = models.BooleanField('Ignore time limit', default=False)
    user_agent = models.CharField(max_length=1024, default='', blank=True)
    qs_params = models.CharField(max_length=1024, default='', blank=True)
    target = models.ForeignKey(Target, verbose_name=_('Target'), related_name='scans', null=True, on_delete=models.CASCADE)
    is_prod = models.BooleanField('Prod', default=False)
    report_ticket = models.CharField(max_length=128, default='', null=True, blank=True)
    send_mail = models.BooleanField('Send Email notification', default=False)
    slug = models.CharField(_('URL Slug'), max_length=1024, default="/")
    scanner_type = models.IntegerField(choices=list(SCANNERS.items()), default=SCANNER_BURP)
    throttle = models.IntegerField(default=0)
    min_ticket_severity = models.IntegerField(choices=list(VulnerabilityType.SEVERITIES.items()),
                                              default=VulnerabilityType.SEVERITY_MEDIUM)
    no_auto_tickets = models.BooleanField('Do not autocreate tickets', default=False)
    users = models.ManyToManyField(settings.AUTH_USER_MODEL, blank=True, related_name="notified_on_scans")
    collaborator_type = models.IntegerField(choices=list(COLLAB_TYPES.items()), default=COLLAB_TYPE_PRIVATE)
    agent_host = models.CharField(_('Agent hostname'), max_length=255, null=True, blank=True)
    scanner_report_url = models.CharField(_('URL'), max_length=1024, null=True, blank=True)

    class Meta:
        verbose_name = _('scan')
        verbose_name_plural = _('scans')
        db_table = 'scans'
        ordering = ['-start']

    @property
    def statusline(self):
        return self.SCAN_STATUSES[self.status]

    @property
    def target_uri(self):
        return self.target.url

    @property
    def public_scanner_report_url(self):
        return self.scanner_report_url.replace(settings.ELLIPTICS_PROXY.get('read'),
                                               settings.ELLIPTICS_PROXY.get('public_read'))

    @property
    def scan_url(self):
        if self.url:
            return self.url
        return self.target.url

    def get_scan_response_stats(self):
        if self.status in [Scan.ST_DONE, Scan.ST_ABORTED]:
            vulns = Vulnerability.objects.select_related('vuln_type').filter(scan=self, vuln_type__name='Stats Issue')
            if not vulns:
                return []
            try:
                data = json.loads(vulns[0].description)
            except Exception:
                return []
            else:
                return data
        return []

    def get_vulnerabilities_count(self, new_only=False, ignore_fp=False):
        total = 0
        if self.status not in [Scan.ST_DONE, Scan.ST_ABORTED]:
            return total
        vulns = Vulnerability.objects.select_related('vuln_type').filter(scan=self)
        for vulnerability in vulns:
            if vulnerability.is_internal:
                continue
            if new_only and (vulnerability.is_triaged or vulnerability.is_false_positive):
                continue
            if ignore_fp and vulnerability.is_false_positive:
                continue
            total += 1
        return total

    vulnerabilities_count = property(get_vulnerabilities_count)

    def get_vulnerabilities(self, new_only=False, ignore_fp=False, show_internal=True):
        vulnerabilities = []
        if self.status in [Scan.ST_DONE, Scan.ST_ABORTED]:
            vulns = Vulnerability.objects.select_related('vuln_type').filter(scan=self)
            for vulnerability in vulns:
                if new_only and (vulnerability.is_triaged or vulnerability.is_false_positive):
                    continue
                if ignore_fp and vulnerability.is_false_positive:
                    continue
                vuln_info = {
                    'summary': vulnerability.vuln_type.summary,
                    'plugin': vulnerability.vuln_type.name,
                    'description': vulnerability.description,
                    'is_false_positive': vulnerability.is_false_positive,
                    'is_triaged': vulnerability.is_triaged,
                    'severity': vulnerability.yseverity,
                    'severity_level': vulnerability.severity,
                    'user_friendly_description': vulnerability.user_friendly_description
                }
                if show_internal:
                    vuln_info['is_internal'] = vulnerability.is_internal
                vulnerabilities.append(vuln_info)
        return vulnerabilities

    vulnerabilities_list = property(get_vulnerabilities)

    @property
    def is_vulnerable(self):
        if self.get_vulnerabilities_count(new_only=True):
            return True
        return False

    def _check_sensitive_vulns(self, severity=VulnerabilityType.SEVERITY_MEDIUM):
        qs = Vulnerability.objects.filter(scan=self, vuln_type__severity__gte=severity).prefetch_related("tracker_tickets")
        for item in qs:
            if item.is_triaged:
                continue
            if not item.is_false_positive:
                return True
        return False

    has_sensitive_vulns = property(_check_sensitive_vulns)

    @property
    def max_severity(self):
        qs = Vulnerability.objects.select_related('vuln_type').filter(scan=self).order_by('-vuln_type__severity')[:1]
        if not qs:
            return 0
        return qs[0].vuln_type.severity

    def gen_burp_scan_profile(self, report_path):
        scan_task = self.scan_task
        profiles_tasks = ProfilesTasks.objects.filter(scan_task=scan_task,
                                                      scan_profile__isnull=False).order_by('id')
        if not profiles_tasks:
            profiles_tasks = ProfilesTargets.objects.filter(target=scan_task.target,
                                                            scan_profile__isnull=False).order_by('id')
        if not profiles_tasks:
            logger.error('No profiles found')
            return ''

        try:
            config = json.loads(profiles_tasks[0].scan_profile.burp_profile)
        except Exception:
            return ''

        profile_fh = NamedTemporaryFile(dir=report_path, delete=False, suffix='.json')

        active_scanner_config = config.get("burp-active-scanner", {})
        active_scanner_config['initial_url'] = self.scan_url

        if self.auth_profile:
            active_scanner_config["auth"] = self.auth_profile.get_yandex_burp_profile(self.scan_url)

        if self.qs_params:
            active_scanner_config["qs_parameters"] = self.qs_params

        if self.user_agent:
            active_scanner_config["user_agent"] = self.user_agent

        if self.throttle > 0:
            active_scanner_config["throttle"] = self.throttle

        if self.collaborator_type == self.COLLAB_TYPE_PRIVATE:
            active_scanner_config["collaborator_server"] = {"location": settings.COLLABORATOR_SERVER,
                                                            "poll_over_unencrypted_http": True,
                                                            "polling_location": "",
                                                            "type": "private"}
        else:
            active_scanner_config["collaborator_server"] = {"location": "",
                                                            "poll_over_unencrypted_http": False,
                                                            "polling_location": "",
                                                            "type": "public"}

        config['burp-active-scanner'] = active_scanner_config
        config['scan_uid'] = '{scan_uid}'
        with open(profile_fh.name, 'w') as profile:
            json.dump(config, profile)
        return profile_fh.name

    def __str__(self):
        return '%s' % (self.scan_task, )


def set_scan_meta(sender, **kwargs):
    scan = kwargs.get('instance')  # type: Scan
    if kwargs.get('created'):
        scan.uid = str(uuid.uuid4())
        scan.slug = slugify_target_url(scan.url)
        scan.save()
        log_scan_to_splunk(scan)


models.signals.post_save.connect(set_scan_meta, sender=Scan)


class VulnTicket(models.Model):
    NON_TRIAGING_RESOLUTIONS = ['tested', 'testing in trunk', 'tested in trunk', 'fixed']
    TRIAGING_RESOLUTIONS = ["invalid", "duplicate", "won'tfix", "willnotfix"]

    TT_ST = 1
    TT_JIRA = 2
    TRACKER_TYPES = {
        TT_ST: 'StarTrack',
        TT_JIRA: 'Jira'
    }
    ST_UNKN = 0
    ST_OPEN = 1
    ST_PROGR = 2
    ST_CLOSED = 3
    TRACKER_STATUS = {
        ST_UNKN: 'Other',
        ST_OPEN: 'Open',
        ST_PROGR: 'In progress',
        ST_CLOSED: 'Closed'
    }
    tracker_type = models.IntegerField(choices=list(TRACKER_TYPES.items()), default=TT_ST)
    ticket_id = models.CharField(max_length=128)
    ticket_status = models.IntegerField(choices=list(TRACKER_STATUS.items()), default=ST_OPEN)
    last_updated = models.DateTimeField(null=True, auto_now=True)
    resolution = models.CharField(max_length=1024, default='')
    webhook_url = models.CharField(max_length=1024, default='')

    objects = models.Manager()
    # we can't name it current_objects b/c django will handle it as default manager
    triaging = TriagedTicketsManager()

    class Meta:
        verbose_name = _('VulnTicket')
        verbose_name_plural = _('VulnTickets')
        db_table = 'w3af_webui_vulnticket'

    def set_status(self, status, resolution=""):
        if status == self.ST_CLOSED:
            st = Startrek(useragent=settings.ST_USER_AGENT, base_url=settings.ST_URL, token=settings.ST_OAUTH_TOKEN)
            try:
                issue = st.issues[self.ticket_id]
                transition = issue.transitions['close']
                transition.execute(
                    resolution=resolution
                )
                for wh in issue.webhooks:
                    wh.delete()
            except Exception as e:
                catch_error({"ticket_id": self.ticket_id})
                logger.error('Ticket close error (%s)' % str(e))
                return
        return

    def update_status(self, status=None, resolution=None):
        if status is not None:
            self.ticket_status = status
        if resolution is not None:
            self.resolution = resolution
        self.save()

    def get_status(self, status):
        if status.lower() == 'open':
            return VulnTicket.ST_OPEN
        if status.lower() == 'closed':
            return VulnTicket.ST_CLOSED
        return VulnTicket.ST_PROGR

    @property
    def is_triaging(self):
        if self.resolution in VulnTicket.NON_TRIAGING_RESOLUTIONS:
            return False
        if self.ticket_status == VulnTicket.ST_CLOSED and \
                self.resolution not in VulnTicket.TRIAGING_RESOLUTIONS:
            return False
        #XXX: see MOLLY-356, bring it back ASAP
        if not self.webhook_url:
            return False
        return True


def handle_ticket_status(sender, **kwargs):
    vuln_ticket = kwargs.get('instance')  # type: VulnTicket
    if vuln_ticket.ticket_status == VulnTicket.ST_CLOSED:
        if vuln_ticket.resolution in ["won'tfix", "willnotfix", "invalid"]:
            for vuln in vuln_ticket.vulnerabilities.all():
                vuln.mark_as_fp()
    if vuln_ticket.ticket_status == VulnTicket.ST_OPEN:
        if vuln_ticket.resolution == "reopened":
            for vuln in vuln_ticket.vulnerabilities.all():
                vuln.unmark_fp()


models.signals.post_save.connect(handle_ticket_status, sender=VulnTicket)


class FalsePositive(models.Model):
    target = models.ForeignKey(Target, verbose_name=_('Target'), on_delete=models.CASCADE)
    vuln_type = models.ForeignKey(VulnerabilityType, verbose_name=_('Vulnerability type'), on_delete=models.CASCADE)
    http_details = models.ManyToManyField(HTTPTransaction, blank=True)
    deleted = models.BooleanField(default=False)
    last_updated = models.DateTimeField(null=True, auto_now=True)
    user = models.ForeignKey(settings.AUTH_USER_MODEL, verbose_name=_('User'), blank=True, null=True, on_delete=models.CASCADE)

    class Meta:
        verbose_name = _('False Positive')
        verbose_name_plural = _('False Positives')
        db_table = 'w3af_webui_falsepositive'


def set_auth_profile_uid(sender, **kwargs):
    if kwargs['created']:
        auth_profile = kwargs['instance']  # type: AuthProfile
        auth_profile.uid = str(uuid.uuid4())
        auth_profile.save()


models.signals.post_save.connect(set_auth_profile_uid, sender=AuthProfile)


class Vulnerability(models.Model):
    scan = models.ForeignKey(Scan, verbose_name=_('Scan'), related_name="vulnerabilities", on_delete=models.CASCADE)
    vuln_type = models.ForeignKey(VulnerabilityType, verbose_name=_('Vulnerability type'), null=True, on_delete=models.CASCADE)
    description = models.TextField(_('Description'), blank=True, null=True)
    http_transaction = models.TextField(_('HTTP Transaction'), blank=True, null=True)
    false_positive = models.ForeignKey(FalsePositive, verbose_name=_('False Positive details'),
                                       null=True, blank=True, on_delete=models.CASCADE)
    http_details = models.ManyToManyField(HTTPTransaction, blank=True)
    tracker_tickets = models.ManyToManyField(VulnTicket, blank=True, related_name="vulnerabilities")
    uid = models.CharField(_('Vuln uid'), blank=True, null=True, max_length=64)
    hidden = models.BooleanField('Silently ignore this vuln', default=False)

    def __str__(self):
        return '%s' % self.vuln_type

    class Meta:
        verbose_name = _('Vulnerability')
        verbose_name_plural = _('Vulnerabilities')
        db_table = 'vulnerabilities'

    @property
    def non_informative(self):
        if self.vuln_type.severity == VulnerabilityType.SEVERITY_IGNORE:
            return True
        return False

    @property
    def is_internal(self):
        return self.vuln_type.is_internal

    @property
    def yseverity(self):
        return VulnerabilityType.SEVERITIES.get(self.vuln_type.severity)

    @property
    def severity(self):
        return self.vuln_type.severity

    def _was_marked_as_fp(self):
        if self.vuln_type.is_internal:
            return False

        if self.false_positive and not self.false_positive.deleted:
            return True

        target = self.scan.target
        if target and target.parent:
            target = target.parent

        if FalsePositiveByVulnType.objects.filter(target=target,
                                                  vuln_type=self.vuln_type,
                                                  deleted=False).exists():
            return True
        return False

    is_false_positive = property(_was_marked_as_fp)

    def was_marked_as_fp_in_past(self):
        if self.vuln_type.is_internal:
            return False

        parent_fps = []
        target = self.scan.target
        if target and target.parent:
            parent_fps = FalsePositive.objects.filter(target=target.parent,
                                                      vuln_type=self.vuln_type).select_related()

        target_fps = FalsePositive.objects.filter(target=target,
                                                  vuln_type=self.vuln_type).select_related()

        if not target_fps and not parent_fps:
            return False

        for http_details in self.http_details.all():
            req_headers = dict()
            resp_headers = dict()

            for item in http_details.request_headers.all():
                req_headers[item.name.lower()] = item.value

            for item in http_details.response_headers.all():
                resp_headers[item.name.lower()] = item.value

            for fp in target_fps:
                for fp_http_details in fp.http_details.all():
                    if self._cmp_http_details(self.vuln_type,
                                              http_details,
                                              fp_http_details,
                                              req_headers,
                                              resp_headers):
                        return True

            for fp in parent_fps:
                for fp_http_details in fp.http_details.all():
                    if self._cmp_http_details(self.vuln_type,
                                              http_details,
                                              fp_http_details,
                                              req_headers,
                                              resp_headers):
                        return True
        return False

    def mark_vulntype_as_fp(self, user=None):
        if self.vuln_type.is_internal:
            return

        target = self.scan.target
        if not target:
            target = self.scan.scan_task.target

        if target.parent:
            target = target.parent

        fp, created = FalsePositiveByVulnType.objects.get_or_create(target=target, vuln_type=self.vuln_type)
        if created and user is not None:
            fp.user = user
            fp.save()

        if fp.deleted:
            fp.deleted = False
            fp.save()

    def mark_as_fp(self, user=None):
        if self.vuln_type.is_internal:
            return

        target = self.scan.target
        if not target:
            target = self.scan.scan_task.target

        if target.parent:
            target = target.parent

        fp = FalsePositive.objects.filter(target=target, vuln_type=self.vuln_type)[:1]
        if fp:
            fp = fp[0]

        if not fp:
            fp = FalsePositive(target=target, vuln_type=self.vuln_type)
            if user:
                fp.user = user
            fp.save()

        for http_details in self.http_details.all():
            found = False
            req_headers = dict()
            resp_headers = dict()

            for item in http_details.request_headers.all():
                req_headers[item.name.lower()] = item.value

            for item in http_details.response_headers.all():
                resp_headers[item.name.lower()] = item.value

            for fp_http_details in fp.http_details.all():
                if self._cmp_http_details(self.vuln_type,
                                          http_details,
                                          fp_http_details,
                                          req_headers,
                                          resp_headers):
                    found = True
                    break
            if found:
                continue
            fp.http_details.add(http_details)
        if not fp:
            return
        
        self.false_positive = fp
        self.save()

        for ticket in self.tracker_tickets.all():
            ticket.set_status(VulnTicket.ST_CLOSED, "willNotFix")

    def _cmp_request_line_paths(self, l1, p2):
        try:
            urp1 = urlparse(l1.split(' ')[1])
            urp2 = urlparse(p2.split(' ')[1])
        except Exception:
            return True
        else:
            if urp1.path == urp2.path:
                return True
        return False

    def unmark_fp(self):
        if self.vuln_type.is_internal:
            return

        if not self.false_positive:
            return

        fp = self.false_positive
        for http_details in self.http_details.all():
            req_headers = dict()
            resp_headers = dict()

            for item in http_details.request_headers.all():
                req_headers[item.name.lower()] = item.value

            for item in http_details.response_headers.all():
                resp_headers[item.name.lower()] = item.value

            for fp_http_details in fp.http_details.all():
                if self._cmp_http_details(self.vuln_type,
                                          http_details,
                                          fp_http_details,
                                          req_headers,
                                          resp_headers):
                    fp.http_details.remove(fp_http_details)
                    break
        self.false_positive = None
        self.save()

    def _cmp_http_details(self, vuln_type, http_details, fp_http_details,
                          req_headers, resp_headers):
        # todo-ezaitov: select most useful headers and add here
        MEANINGFUL_HEADERS = []

        if fp_http_details.status_line != http_details.status_line:
            return False

        if not self._cmp_request_line_paths(fp_http_details.request_line,
                                            http_details.request_line):
            return False

        for h in fp_http_details.request_headers.all():
            if h.name.lower() not in MEANINGFUL_HEADERS:
                continue
            if h.value != req_headers.get(h.name.lower(), ''):
                return False
        for h in fp_http_details.response_headers.all():
            if h.name.lower() not in MEANINGFUL_HEADERS:
                continue
            if h.value != resp_headers.get(h.name.lower(), ''):
                return False
        return True

    def was_triaged(self):
        for ticket in self.tracker_tickets.all():
            if ticket.is_triaging:
                return True
        return False

    def was_triaged_in_past(self):
        if self.vuln_type.is_internal:
            return True

        if self.was_triaged():
            return True

        max_past_value = datetime.now() + relativedelta(days=-90)
        parent_vulns = []
        if self.scan.target.parent:
            parent_vulns = Vulnerability.objects.filter(
                                scan__target=self.scan.target.parent,
                                vuln_type=self.vuln_type, scan__finish__gte=max_past_value)\
                .exclude(tracker_tickets=None)

        other_vulns = Vulnerability.objects.filter(
                            scan__target=self.scan.target,
                            vuln_type=self.vuln_type, scan__finish__gte=max_past_value)\
            .exclude(tracker_tickets=None)

        triaging = False
        for http_details in self.http_details.all():
            req_headers = dict()
            resp_headers = dict()

            for item in http_details.request_headers.all():
                req_headers[item.name.lower()] = item.value

            for item in http_details.response_headers.all():
                resp_headers[item.name.lower()] = item.value

            for vuln in parent_vulns:
                if vuln.is_false_positive:
                    continue
                if vuln.tracker_tickets.count() == 0:
                    continue
                for vuln_http_details in vuln.http_details.all():
                    if self._cmp_http_details(self.vuln_type,
                                              http_details,
                                              vuln_http_details,
                                              req_headers,
                                              resp_headers):
                        for ticket in vuln.tracker_tickets.all():
                            if ticket.is_triaging:
                                self.tracker_tickets.add(ticket)
                                # XXX: can be slooow
                                triaging = True

            for vuln in other_vulns:
                if vuln.is_false_positive:
                    continue
                if vuln.tracker_tickets.count() == 0:
                    continue
                for vuln_http_details in vuln.http_details.all():
                    if self._cmp_http_details(self.vuln_type,
                                              http_details,
                                              vuln_http_details,
                                              req_headers,
                                              resp_headers):
                        for ticket in vuln.tracker_tickets.all():
                            if ticket.is_triaging:
                                self.tracker_tickets.add(ticket)
                                # XXX: can be slooow
                                triaging = True

        return triaging

    is_triaged = property(was_triaged)

    @property
    def vulnerable_host(self):
        p = urlparse(self.scan.url)
        return urlunparse((p[0], p[1], '', '', '', ''))

    @property
    def user_friendly_description(self):
        if self.vuln_type.name == 'Stats Issue':
            stats = self.scan.get_scan_response_stats()
            if not stats:
                return ''
            res = '#|\n|| **Method** | **Response code** | **Count** ||\n'
            for item in stats:
                method = item.get('method')
                for resp in item.get("responses"):
                    count = resp.get("count", 0)
                    status_code = resp.get("status_code")
                    res += '|| {} | {} | {} ||\n'.format(method, status_code, count)
            res += '|#'
            return res
        t = Template(('{% spaceless %}{% load sanitizer %}{% autoescape off %}'
                      '{% if vuln.vuln_type.description %}{{ vuln.vuln_type.wiki_description }}\n{% endif %}'
                      '{{ vuln.wiki_description|strip_html }}\n'
                      '{% endautoescape %}{% endspaceless %}'))
        d = {"vuln": self}
        return t.render(Context(d))

    @property
    def human_name(self):
        return self.vuln_type.human_name

    @property
    def wiki_description(self):
        if not self.description:
            return ''
        description = re.sub(r'<b>([\s+]?)', '**', self.description)
        description = re.sub(r'([\s+]?)</b>', '**', description)
        description = re.sub(r'<strong>([\s+]?)', '**', description)
        description = re.sub(r'([\s+]?)</strong>', '**', description)
        description = description.replace('<br>', '\n')
        description = description.replace('<ul>', '\n')
        description = description.replace('</ul>', '\n')
        description = description.replace('<li>', '* ')
        description = description.replace('</li>', '')
        description = description.replace('<pre>', '%%')
        description = description.replace('</pre>', '%%')
        return description


def set_vuln_uid(sender, **kwargs):
    if kwargs.get('created'):
        vuln = kwargs.get('instance')  # type: Vulnerability
        vuln.uid = str(uuid.uuid4())
        vuln.save()


models.signals.post_save.connect(set_vuln_uid, sender=Vulnerability)


class ISNotification(models.Model):
    STATUS_OPEN = 1
    STATUS_CLOSED = 2
    NOTIFICATION_STATUS = {
        STATUS_OPEN: 'Open',
        STATUS_CLOSED: 'Closed'
    }
    scan = models.ForeignKey(Scan, verbose_name=_('Scan'), on_delete=models.CASCADE)
    vuln = models.ForeignKey(Vulnerability, verbose_name=_('Vulnerability'), null=True, on_delete=models.CASCADE)
    status = models.IntegerField(choices=list(NOTIFICATION_STATUS.items()), default=STATUS_OPEN)
    user = models.ForeignKey(settings.AUTH_USER_MODEL, verbose_name=_('User'), blank=True, null=True, on_delete=models.CASCADE)
    description = models.TextField(_('Description'), blank=True, null=True)


class TargetUriMap(models.Model):
    uid = models.CharField(_('Uid'), max_length=128, blank=False)
    name = models.CharField(_('Name'), max_length=128, blank=False)
    map = models.TextField(_('URI list'), blank=True, null=True, max_length=2048, help_text='One per line')
    comment = models.TextField(_('Description'), blank=True, null=True, max_length=500)
    users = models.ManyToManyField(settings.AUTH_USER_MODEL, blank=True)

    class Meta:
        verbose_name = _('URI map')
        verbose_name_plural = _('URI maps')
        db_table = 'w3af_webui_targeturimap'

    def __str__(self):
        return '%s (%s)' % (self.name, self.uid)


def set_map_uid(sender, **kwargs):
    if kwargs.get('created'):
        uri_map = kwargs.get('instance')  # type: TargetUriMap
        uri_map.uid = str(uuid.uuid4())
        uri_map.save()


models.signals.post_save.connect(set_auth_profile_uid, sender=TargetUriMap)


class CrasherStatus(models.Model):
    STATUS_IDLE = 0
    STATUS_RUNNING = 1
    STATUS_NEEDRUN = 2
    SCAN_STATUS = {
        STATUS_IDLE: 'Not running',
        STATUS_RUNNING: 'Running',
        STATUS_NEEDRUN: 'Waiting for start',
    }
    status = models.IntegerField(choices=list(SCAN_STATUS.items()), default=STATUS_IDLE)
    user = models.ForeignKey(settings.AUTH_USER_MODEL, verbose_name=_('User'), blank=True, null=True, on_delete=models.CASCADE)
    last_updated = models.DateTimeField(null=True, auto_now=True)

    class Meta:
        verbose_name = _('CrasherStatus')
        verbose_name_plural = _('Crasher Statuses')
        db_table = 'w3af_webui_crasherstatus'


def set_target_slug(sender, **kwargs):
    target = kwargs.get('instance')  # type: Target
    target.slug = slugify_target_url(target.url)


models.signals.pre_save.connect(set_target_slug, sender=Target)


class CrasherChunk(models.Model):
    uid = models.CharField(_('Uid'), max_length=128, blank=False)
    data = models.TextField(_('Data'), blank=True, null=True)

    class Meta:
        verbose_name = _('Crasher Chunk')
        verbose_name_plural = _('Crasher Chunks')
        db_table = 'w3af_webui_crasherchunk'


# todo-ezaitov: refactor
def set_chunk_uid(sender, **kwargs):
    target = kwargs.get('instance')  # type: CrasherChunk
    target.uid = str(uuid.uuid4())


models.signals.pre_save.connect(set_chunk_uid, sender=CrasherChunk)
