import requests
import settings
import traceback
import json


class ErrorTracer(object):
    def __init__(self, url):
        self.url = url
        self.session = None

    def init_session(self):
        self.session = requests.Session()
        retry_adapter = requests.adapters.HTTPAdapter(max_retries=3)
        
        self.session.mount("http://", retry_adapter)
        self.session.mount("https://", retry_adapter)
        
    def try_send_trace(self, stacktrace, context=None):
        """Try to send trace, no throw exception if error"""

        obj = dict(stacktrace=stacktrace)

        if context is not None:
            try:
                obj["context"] = json.dumps(context)
            except:
                pass

        try:
            self.session.post(self.url, json=obj)
        except:
            pass


error_tracer = None


def init_global_error_tracer():
    global error_tracer
    error_tracer = ErrorTracer(settings.ERRORTRACER_URL)
    error_tracer.init_session()


def catch_error(context=None):
    """Catch current exception, if context passed send it with stacktrace"""
    global error_tracer
    if error_tracer is None:
        init_global_error_tracer()

    try:
        error_tracer.try_send_trace(
            traceback.format_exc(),
            context=context
        )
    except:
        # handle exception in traceback.format_exc
        pass