# -*- coding: utf-8 -*-


import struct
import subprocess
import os
import socket
import json
import urllib.parse
import stat
import time
import tempfile
import zlib
import hashlib
import traceback
import uuid
import shlex

import porto
import celery
import requests

from .models import RequestSamples
from .utils import is_resolvable, killpstree, upload_to_elliptics, quote_commands, join_envs

from logging import getLogger


logger = getLogger(__name__)


class RemoteBurpTask(celery.Task):
    soft_time_limit = 7200
    name = "RemoteBurpTask"
    default_max_concurrent = struct.pack(">i", 637)

    @staticmethod
    def common_project():
        """
        Using lazy getter, because not all application, 
        which use this file need loading project file.
        """
        hidden_attr = "_common_project"
        if hasattr(RemoteBurpTask, hidden_attr):
            return getattr(RemoteBurpTask, hidden_attr)
        project = open("/usr/lib/yandex/burp2/common_project", "rb").read()
        setattr(RemoteBurpTask, hidden_attr, project)
        return project

    def run(
        self,
        text_config,
        work_dir,
        profile_file,
        req_samples_url="",
        sample_format=RequestSamples.FMT_JSON,
        ignore_time_limit=False,
    ):
        self.hostname = socket.gethostname()
        report_tpl = os.path.splitext(profile_file)[0]
        self.scan_uid = None

        if not os.path.isdir(work_dir):
            os.makedirs(work_dir)
            os.chmod(work_dir, 0o777)

        try:
            config = json.loads(text_config)
            config = self._from_old_config(config)
        except Exception:
            traceback.print_exc()
            return self._make_error("<json config error>")

        self.scan_uid = config.get("scan_uid")
        active_scanner_config = config.get("burp-active-scanner", {})
        self._set_state("RECEIVED")

        initial_url = active_scanner_config.get("target", {}).get("initial_url")

        parsed_initial_url = urllib.parse.urlparse(initial_url)
        if not parsed_initial_url.netloc:
            return self._make_error("<url format error>")

        if not is_resolvable(initial_url):
            return self._make_error("<no such host>")

        try:
            requests.get(initial_url, timeout=5, allow_redirects=False, verify=False)
        except Exception:
            return self._make_error("<target unavailable>")

        report_path = report_tpl + "_report.xml"
        self.has_key(active_scanner_config, "scan")
        active_scanner_config["scan"]["report_path"] = report_path
        scan_log_path = report_tpl + "_http.log"
        active_scanner_config["scan_log_path"] = scan_log_path
        sitemap_path = report_tpl + "_sitemap.json"
        self.has_key(active_scanner_config, "target")
        active_scanner_config["target"]["sitemap_path"] = sitemap_path

        config["burp-active-scanner"] = active_scanner_config
        with open(profile_file, "wb") as configfile:
            configfile.write(json.dumps(config))

        os.chmod(profile_file, stat.S_IRUSR | stat.S_IRGRP | stat.S_IROTH)

        env = os.environ.copy()
        env["MOLLY_CONFIG"] = profile_file
        proto_envs = join_envs(env)

        proxy_port = self._get_random_port()
        project_options_path = report_tpl + "_project_options.json"
        self._prepare_project_options(
            project_options_path, proxy_port, active_scanner_config
        )

        self._set_state("STARTED")

        project_file = self._prepare_project(10)

        crawl_burp_stdout = open(report_tpl + "_crawl_stdout.txt", "wb")
        crawl_burp_stderr = open(report_tpl + "_crawl_stderr.txt", "wb")

        child = None
        crawl_args = [
            "/usr/local/jdk-13/bin/java",
            "-jar",
            "-Xmx2048m",
            # "-Djava.net.preferIPv6Addresses=true", not work with chrome
            "-Djava.awt.headless=true",
            "/usr/lib/yandex/burp2/burpsuite_pro.jar",
            "--start-crawling",
            "--unpause-spider-and-scanner",
            "--user-config-file=/usr/lib/yandex/burp2/user_options.json",
            "--project-file={}".format(project_file.name),
            "--config-file={}".format(project_options_path),
        ]
        crawl_command = quote_commands(crawl_args)

        crawl_timeout = self._get_timeout(active_scanner_config, "crawl")
        crawl_return_code = None
        try:
            conn = porto.Connection()
            child = conn.CreateWeakContainer("self/child-{}".format(uuid.uuid4()))
            child.SetProperty("enable_porto", "false")
            child.SetProperty("isolate", "true")
            child.SetProperty("stdout_path", "/dev/fd/{}".format(crawl_burp_stdout.fileno()))
            child.SetProperty("stderr_path", "/dev/fd/{}".format(crawl_burp_stderr.fileno()))
            child.SetProperty("command", crawl_command)
            child.SetProperty("env", proto_envs)
            child.Start()
            child.Wait(timeout=crawl_timeout*1000)
            crawl_return_code = int(child.GetProperty("exit_code"))
        except:
            import traceback; traceback.print_exc()
            return self._make_error("<report read error>", aborted=True)
        finally:
            if child is not None:
                child.Destroy()
                child = None
        
        project_file.close()
        crawl_burp_stdout.close()
        crawl_burp_stderr.close()

        if crawl_return_code is None:
            logger.error("No scan return code, some porto err")
            return self._make_error("<report read error>", aborted=True)

        if crawl_return_code != 0:
            logger.error("Crawling return code: %d" % crawl_return_code)
            return self._make_error("<report read error>", aborted=True)


        project_file = self._prepare_project(10)

        scan_burp_stdout = open(report_tpl + "_scan_stdout.txt", "wb")
        scan_burp_stderr = open(report_tpl + "_scan_stdout.txt", "wb")

        scan_args = [
                "/usr/local/jdk-13/bin/java",
                "-jar",
                "-Xmx2048m",
                # "-Djava.net.preferIPv6Addresses=true", not work with chrome
                "-Djava.awt.headless=true",
                "/usr/lib/yandex/burp2/burpsuite_pro.jar",
                "--start-scanning",
                "--unpause-spider-and-scanner",
                "--user-config-file=/usr/lib/yandex/burp2/user_options.json",
                "--project-file={}".format(project_file.name),
                "--config-file={}".format(project_options_path),
        ]
        scan_command = quote_commands(scan_args)

        scan_timeout = self._get_timeout(active_scanner_config, "scan")
        scan_return_code = None
        try:
            conn = porto.Connection()
            child = conn.CreateWeakContainer("self/child-{}".format(uuid.uuid4()))
            child.SetProperty("enable_porto", "false")
            child.SetProperty("isolate", "true")
            child.SetProperty("stdout_path", "/dev/fd/{}".format(scan_burp_stdout.fileno()))
            child.SetProperty("stderr_path", "/dev/fd/{}".format(scan_burp_stderr.fileno()))
            child.SetProperty("command", scan_command)
            child.SetProperty("env", proto_envs)
            child.Start()
            child.Wait(timeout=scan_timeout*1000)
            scan_return_code = int(child.GetProperty("exit_code"))
        except:
            import traceback; traceback.print_exc()
            return self._make_error("<report read error>", aborted=True)
        finally:
            if child is not None:
                child.Destroy()
        

        project_file.close()
        scan_burp_stdout.close()
        scan_burp_stderr.close()

        if scan_return_code is None:
            logger.error("No scan return code, some porto err")
            return self._make_error("<report read error>", aborted=True)

        if scan_return_code != 0:
            logger.error("Scanning return code: %d" % scan_return_code)
            return self._make_error("<report read error>", aborted=True)

        try:
            scan_report = open(report_path, "rb").read()
        except IOError:
            return self._make_error("<report read error>", aborted=True)
        
        max_report = 150000000
        if len(scan_report) > max_report:
            scan_report = scan_report[:max_report]
            idx = scan_report.rfind(">")
            scan_report = scan_report[: idx + 1]
            scan_report += "</issues>"

        scan_log_url = ""
        try:
            if os.path.isfile(scan_log_path):
                scan_log_url = upload_to_elliptics(
                    hashlib.sha256(config.get("scan_uid")).hexdigest(),
                    open(scan_log_path, "rb").read(),
                    public=True,
                    ttl=80,
                )
        except Exception:
            # we remove logs just after upload b/c they're too big to store on agents
            os.unlink(scan_log_path)
            pass

        scan_report_url = ""
        try:
            compressed_scan_report = zlib.compress(scan_report)
            scan_report_url = upload_to_elliptics(
                hashlib.sha256(compressed_scan_report).hexdigest(),
                compressed_scan_report,
                ttl=7,
            )
        except Exception:
            pass

        self._set_state("SUCCESS")
        return scan_report_url, scan_log_url, False, self.hostname

    def _get_timeout(self, config, action):
        timeout = config.get(action, {}).get("timeout", 0) + 60
        # TODO(ilyaon): move minimal timout to config
        if action == "scan":
            default_timeout = 600
        elif action == "crawl":
            default_timeout = 300
        timeout = max(timeout, default_timeout)
        return timeout

    def _get_random_port(self):
        s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        s.bind(("", 0))
        s.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        port = s.getsockname()[1]
        s.close()
        return port

    def _prepare_project(self, max_concurrent):
        """Patch project and return new project file"""
        value = struct.pack(">i", max_concurrent)
        project_bytes = self.common_project().replace(
            self.default_max_concurrent, value
        )
        project_temp = tempfile.NamedTemporaryFile()
        project_temp.write(project_bytes)
        project_temp.flush()
        return project_temp

    def _prepare_project_options(
        self, project_options_path, proxy_port, active_scanner_config
    ):
        try:
            project_options = open(project_options_path, "w")
            with open("/usr/lib/yandex/burp2/project_options.json", "r") as fd:
                burp_config = json.load(fd)
                burp_config["proxy"]["request_listeners"] = []
                burp_config["proxy"]["request_listeners"].append(
                    {
                        "certificate_mode": "per_host",
                        "listen_mode": "loopback_only",
                        "listener_port": proxy_port,
                        "running": True,
                    }
                )
                if active_scanner_config.get("collaborator_server"):
                    misc_config = burp_config["project_options"].get("misc", {})
                    misc_config["collaborator_server"] = active_scanner_config.get(
                        "collaborator_server"
                    )
                    burp_config["project_options"]["misc"] = misc_config
                json.dump(burp_config, project_options)
                project_options.close()
            return None
        except Exception:
            traceback.print_exc()
            return None

    def _make_error(self, description, aborted=False):
        self._set_state("FAILURE")
        scan_log_url = ""
        return description, scan_log_url, aborted, self.hostname

    def _set_state(self, state):
        self.update_state(
            state=state, meta={"uid": self.scan_uid, "worker": self.hostname}
        )

    def _from_old_config(self, config):
        """ Transform config format from old version to new"""
        active_scanner = config.get("burp-active-scanner")

        initial_url = active_scanner.get("initial_url")
        if initial_url is not None:
            self.has_key(active_scanner, "target")
            active_scanner["target"]["initial_url"] = initial_url
            del active_scanner["initial_url"]

        qs_parameters = active_scanner.get("qs_parameters")
        if qs_parameters is not None:
            self.has_key(active_scanner, "modifier")
            active_scanner["modifier"]["qs_parameters"] = qs_parameters
            del active_scanner["qs_parameters"]

        user_agent = active_scanner.get("user_agent")
        if user_agent is not None:
            self.has_key(active_scanner, "modifier")
            active_scanner["modifier"]["user_agent"] = user_agent
            del active_scanner["user_agent"]

        return config

    @staticmethod
    def has_key(obj, key):
        if key not in obj:
            obj[key] = dict()