# -*- coding: utf-8 -*-

import os
import json
import operator
import urllib.parse
import re
import datetime

import logging

from django import http
from django.template.loader import render_to_string
from django.shortcuts import HttpResponse
from django.http import HttpResponseNotFound, HttpResponseRedirect
from django.shortcuts import render, get_object_or_404, redirect
from django.utils.text import slugify
from django.contrib.auth.decorators import login_required, user_passes_test
from django.conf import settings
from django.contrib import messages
from django.urls import reverse
from django.views.generic import TemplateView, ListView, View
from django.utils.decorators import method_decorator
from django.db.models import Q
from django.core.exceptions import PermissionDenied

from startrek_client import Startrek

from .models import ScanTask, Scan, Vulnerability, VulnerabilityType, Target, VulnTicket, AuthProfile, \
    ISNotification, TargetUriMap, ScanProfile, ProfilesTasks, CrasherStatus
from .forms import AuthProfileForm, TargetUriMapForm, NewScanForm, TargetMergeForm, \
    AntirobotWizardForm, CrasherStatusForm, TargetEditForm, ScanTargetEditForm, NewTargetForm

from .celery_app import app

from .tasks import run_scan, stop_remote_scan
from .utils import get_oauth_token, get_st_queues_from_abc_by_username, \
    get_admins_from_golem_by_uri, slugify_target_url, get_st_queue_by_abc_id, get_abc_members, \
    get_abc_scopes, read_from_mds

from .notification.send_mail import send_message
from functools import reduce

logger = logging.getLogger(__name__)


class StTicketView(View):
    def post(self, request, *args, **kwargs):
        vuln = get_object_or_404(Vulnerability, uid=request.POST.get('vuln_id', 'none'))
        resp = {'status': 'unknown'}
        st_queue = request.POST.get('st_queue', '')
        # users post full URLS sometimes
        ticket_id = request.POST.get('ticket_id', '').strip('/')
        if '/' in ticket_id:
            ticket_id = ticket_id.split('/')[-1]

        body = request.POST.get('ticket_body', '')
        summary = request.POST.get('ticket_summary', '')
        st = Startrek(useragent=settings.ST_USER_AGENT, base_url=settings.ST_URL, token=settings.ST_OAUTH_TOKEN)
        if ticket_id:
            vt, created = VulnTicket.objects.get_or_create(tracker_type=VulnTicket.TT_ST, ticket_id=ticket_id)
            if created or not vt.webhook_url:
                vuln.tracker_tickets.add(vt)
                try:
                    issue = st.issues[ticket_id]
                except Exception as e:
                    resp['msg'] = str(e)
                    resp['keys'] = []
                    resp['links'] = ''
                    resp['status'] = 'error'
                    return HttpResponse(json.dumps(resp), content_type='application/json; charset=utf-8', status=500)

                try:
                    hook = issue.webhooks.create(endpoint=settings.APP_URL + reverse('st_webhook_handler'),
                                                 filter={"key": ticket_id})
                except Exception as e:
                    logger.error('Ticket webhook error (%s)' % str(e))
                else:
                    vt.webhook_url = hook.self
                    vt.save()

            vuln.tracker_tickets.add(vt)
            resp['keys'] = [x.ticket_id for x in vuln.tracker_tickets.all()]
            resp['links'] = ['<a href="%s/%s">%s</a>&nbsp;' % (settings.ST_UI_URL, x.ticket_id, x.ticket_id)
                             for x in vuln.tracker_tickets.all()]
            resp['status'] = 'new'
            return HttpResponse(json.dumps(resp), content_type='application/json; charset=utf-8')

        if st_queue:
            issue_queue = st_queue
        else:
            issue_queue = vuln.scan.target.st_queue
        if st_queue and vuln.scan.target.st_queue is None:
            vuln.scan.target.st_queue = st_queue
            vuln.scan.target.save()

        if summary:
            issue_summary = summary
        else:
            issue_summary = vuln.vuln_type.summary

        if body:
            issue_description = body
        else:
            issue_description = vuln.description

        issue_tags = []
        if not request.GET.get("nomolly"):
            issue_tags = ["molly"]

        issue = None
        ticket_id = None
        # not all ST queues have issue type "bug", so we retry w/o it
        error_msg = ''
        for i in range(0, 2):
            issue_type = 'Bug'
            try:
                if i == 1:
                    issue_type = 'Task'
                issue = st.issues.create(
                    queue=issue_queue,
                    type={'name': issue_type},
                    summary=issue_summary,
                    description=issue_description,
                    followers=settings.ST_OAUTH_USERNAMES,
                    tags=issue_tags,
                    createdBy=request.user.username
                )
                ticket_id = issue.key
                break
            except Exception as e:
                error_msg = str(error_msg)
                logger.error('Ticket error (%s)' % str(e))
                continue
        if not issue or not ticket_id:
            resp['keys'] = []
            resp['links'] = ''
            resp['msg'] = error_msg
            resp['status'] = 'error'
            return HttpResponse(json.dumps(resp), content_type='application/json; charset=utf-8')

        severity = 0
        if vuln.vuln_type.tracker_severity:
            severity = vuln.vuln_type.tracker_severity

        issue.update(security='Yes', securitySeverity=str(severity), ignore_version_change=True)
        vt, created = VulnTicket.objects.get_or_create(tracker_type=VulnTicket.TT_ST, ticket_id=ticket_id)
        vuln.tracker_tickets.add(vt)
        if created:
            vuln.tracker_tickets.add(vt)
            try:
                hook = issue.webhooks.create(endpoint=settings.APP_URL + reverse('st_webhook_handler'),
                                             filter={"key": ticket_id})
            except Exception as e:
                logger.error('Ticket webhook error (%s)' % str(e))
            else:
                vt.webhook_url = hook.self
                vt.save()

        resp['keys'] = [x.ticket_id for x in vuln.tracker_tickets.all()]
        resp['links'] = '&nbsp;'.join(['<a href="%s/%s">%s</a>' % (settings.ST_UI_URL, x.ticket_id, x.ticket_id)
                                       for x in vuln.tracker_tickets.all()])
        resp['status'] = 'new'
        return HttpResponse(json.dumps(resp), content_type='application/json; charset=utf-8')

    @method_decorator(login_required)
    def dispatch(self, request, *args, **kwargs):
        return super(StTicketView, self).dispatch(request, *args, **kwargs)


class ReportView(TemplateView):
    template_name = "report_view.html"

    def get(self, request, *args, **kwargs):
        progressive = (request.GET.get('progressive', '') == 'True')
        scan = get_object_or_404(Scan, uid=kwargs.get('scan_uid'))
        if request.user.is_superuser:
            vulns = Vulnerability.objects.filter(scan=scan).order_by('-vuln_type__severity')
        else:
            vulns = Vulnerability.objects.filter(scan=scan,
                                                 vuln_type__severity__gt=VulnerabilityType.SEVERITY_INFORMATION)\
                .order_by('-vuln_type__severity')
        count_total = vulns.filter(vuln_type__is_internal=False).count()
        fp_count = 0
        for item in vulns:
            if item.is_false_positive:
                fp_count += 1
        if scan.show_report_time is None:
            scan.show_report_time = datetime.datetime.now()
            scan.save(update_fields=["show_report_time"])
        triaged_vulns = []
        new_vulns = []
        for vuln in vulns:
            if vuln.is_triaged or vuln.is_false_positive:
                triaged_vulns.append(vuln)
            else:
                new_vulns.append(vuln)
        scan_state = None
        scan_worker = ''
        if scan.task_id:
            result = app.AsyncResult(scan.task_id)
            scan_state = result.state
            job_info = result.info
            if isinstance(job_info, dict):
                scan_worker = job_info.get('worker', '')
            elif isinstance(job_info, tuple) and len(job_info) > 3:
                scan_worker = job_info[3]
        if scan.agent_host:
            scan_worker = scan.agent_host
        return self.render_to_response(self.get_context_data(
            scan=scan,
            state=scan_state,
            worker=scan_worker,
            vulns=vulns,
            triaged_vulns=triaged_vulns,
            new_vulns=new_vulns,
            fp_count=fp_count,
            total_count=count_total,
            show_fp=request.GET.get('show_fp', False),
            progressive=progressive,
        ))

    @method_decorator(login_required)
    def dispatch(self, request, *args, **kwargs):
        return super(ReportView, self).dispatch(request, *args, **kwargs)


class CancelScanView(TemplateView):
    template_name = "report_view.html"

    def post(self, request, *args, **kwargs):
        scan = get_object_or_404(Scan, uid=kwargs.get('scan_uid'))
        stop_remote_scan.delay(scan.id)
        scan.status = Scan.ST_ABORTED
        scan.save()
        return HttpResponseRedirect(reverse('show_report', kwargs={'scan_uid': scan.uid}))

    @method_decorator(login_required)
    def dispatch(self, request, *args, **kwargs):
        return super(CancelScanView, self).dispatch(request, *args, **kwargs)


class VulnFalsePositiveView(View):
    def post(self, request, *args, **kwargs):
        """
        View for ajax request for mark vulnerability as false positive
        """
        vuln_id = request.POST.get('vuln_id', 'uniq')
        vuln = Vulnerability.objects.get(uid=vuln_id)
        try:
            if not vuln.is_false_positive:
                vuln.mark_as_fp(user=request.user)
            else:
                vuln.unmark_fp()

            resp = json.dumps({
                'status': 'ok',
                'is_positive': vuln.is_false_positive,
            })
        except Exception:
            resp = json.dumps({
                'status': 'fail',
            })
        return HttpResponse(resp, content_type='application/json')

    @method_decorator(login_required)
    def dispatch(self, request, *args, **kwargs):
        return super(VulnFalsePositiveView, self).dispatch(request, *args, **kwargs)


class VulnTypeFalsePositiveView(View):
    def post(self, request, *args, **kwargs):
        """
        View for ajax request for mark vulnerability as false positive
        """
        vuln_id = request.POST.get('vuln_id', 'uniq')
        vuln = Vulnerability.objects.get(uid=vuln_id)
        try:
            if not vuln.is_false_positive:
                vuln.mark_vulntype_as_fp(user=request.user)

            resp = json.dumps({
                'status': 'ok',
                'is_positive': vuln.is_false_positive,
            })
        except Exception:
            resp = json.dumps({
                'status': 'fail',
            })
        return HttpResponse(resp, content_type='application/json')

    @method_decorator(login_required)
    def dispatch(self, request, *args, **kwargs):
        return super(VulnTypeFalsePositiveView, self).dispatch(request, *args, **kwargs)


class APITokenView(TemplateView):
    template_name = "token.html"

    def get(self, request, *args, **kwargs):
        token = ''
        code = request.GET.get('code', '')
        if code and re.match('(\d+)', code):
            token = get_oauth_token(code)
        return self.render_to_response(self.get_context_data(token=token,
                                                             client_id=settings.OAUTH_CLIENT_ID))

    @method_decorator(login_required)
    def dispatch(self, request, *args, **kwargs):
        return super(APITokenView, self).dispatch(request, *args, **kwargs)


class AuthProfileView(TemplateView):
    template_name = "auth_profile.html"
    _auth_profile = None
    _user_auth_profiles = []

    def get_context_data(self, **kwargs):
        context = super(AuthProfileView, self).get_context_data(**kwargs)
        context['auth_profiles'] = self._user_auth_profiles
        return context

    def get(self, request, *args, **kwargs):
        if self._auth_profile:
            form = AuthProfileForm(instance=self._auth_profile)
        else:
            form = AuthProfileForm()
        return self.render_to_response(self.get_context_data(form=form))

    def post(self, request, *args, **kwargs):
        if self._auth_profile:
            form = AuthProfileForm(request.POST, request.FILES, instance=self._auth_profile)
        else:
            form = AuthProfileForm(request.POST, request.FILES)
        if form.is_valid():
            self._auth_profile = form.save()
            self._auth_profile.users.add(request.user)
            messages.add_message(request, messages.SUCCESS, "Auth profile saved")
            return redirect('auth_profiles_list')
        else:
            messages.add_message(request, messages.ERROR, "Auth profile saving error")
        return self.render_to_response(self.get_context_data(form=form))

    @method_decorator(login_required)
    def dispatch(self, request, *args, **kwargs):
        if kwargs.get('profile_id'):
            self._auth_profile = get_object_or_404(AuthProfile, id=kwargs.get('profile_id'))
            if request.user not in self._auth_profile.users.all():
                raise http.Http404
        self._user_auth_profiles = AuthProfile.objects.filter(users__in=[request.user.id])
        return super(AuthProfileView, self).dispatch(request, *args, **kwargs)


class ScanListView(ListView):
    model = Scan
    template_name = "scan_list.html"
    paginate_by = 20
    _user = None
    _severity = -1

    def get_context_data(self, **kwargs):
        context = super(ScanListView, self).get_context_data(**kwargs)
        context['crasher'] = self._filter
        context['severity'] = self._severity
        return context

    def get_queryset(self):
        # TODO: optimize!
        filters = []
        if not self._user.is_superuser:
            filters.append(Q(user=self._user))
        filters.append(Q(uid__isnull=False))
        if self._filter == 'prod':
            filters.append(Q(is_prod=True))
        elif self._filter == 'int':
            filters.append(Q(is_prod=False))

        latest_old_scan_time = datetime.datetime.now() + datetime.timedelta(days=-30)
        filters.append(Q(start__gte=latest_old_scan_time))

        qs = super(ScanListView, self).get_queryset().select_related()\
            .filter(reduce(operator.and_, filters)).order_by('-start')
#        qs = qs.distinct('target')
#        for scan in qs:
#            if Vulnerability.objects.filter(scan=scan, vuln_type__severity__gte=self._severity).count() > 0:
#                res.append(scan)
        return qs

    @method_decorator(login_required)
    def dispatch(self, request, *args, **kwargs):
        self._user = request.user
        self._filter = request.GET.get('crasher', 'prod')
        try:
            self._severity = int(request.GET.get('severity', -1))
        except ValueError:
            self._severity = -1
        if self._severity not in list(VulnerabilityType.SEVERITIES.keys()):
            raise http.Http404
        return super(ScanListView, self).dispatch(request, *args, **kwargs)


class HTTPTransactionView(TemplateView):
    template_name = "http_trans.html"

    def get(self, request, *args, **kwargs):
        if not kwargs.get('vuln_id'):
            raise http.Http404

        vuln = get_object_or_404(Vulnerability, uid=kwargs.get('vuln_id'))
        return self.render_to_response(self.get_context_data(vuln=vuln))

    @method_decorator(login_required)
    def dispatch(self, request, *args, **kwargs):
        return super(HTTPTransactionView, self).dispatch(request, *args, **kwargs)


class CallSibView(TemplateView):
    def post(self, request, *args, **kwargs):
        scan = get_object_or_404(Scan, uid=request.POST.get('scan_id'))
        invalid_target = False
        vuln_uid = None
        correct_service = None
        if request.POST.get('vuln_id'):
            vuln = get_object_or_404(Vulnerability, uid=request.POST.get('vuln_id'))
            vuln_uid = vuln.uid
        else:
            invalid_target = True
        if request.POST.get('correct_service'):
            correct_service = request.POST.get('correct_service')
        send_message(to_email=settings.CALLNOTIFICATION_EMAILS,
                     template='notifications/call.html',
                     subject='%s@ is calling you to Molly' % request.user.username,
                     data={'username': request.user.username, 'vuln_uid': vuln_uid,
                           'scan_uid': scan.uid, 'invalid_target': invalid_target,
                           'correct_service': correct_service},
                     sender_email=settings.EMAIL_HOST_USER)
        return HttpResponse(json.dumps({'ok': True}), content_type='application/json; charset=utf-8')


    @method_decorator(login_required)
    def dispatch(self, request, *args, **kwargs):
        return super(CallSibView, self).dispatch(request, *args, **kwargs)


class TargetMapView(TemplateView):
    template_name = "target_map.html"
    _target_map = None
    _user_target_maps = []

    def get_context_data(self, **kwargs):
        context = super(TargetMapView, self).get_context_data(**kwargs)
        context['target_maps'] = self._user_target_maps
        return context

    def get(self, request, *args, **kwargs):
        if self._target_map:
            form = TargetUriMapForm(instance=self._target_map)
        else:
            form = TargetUriMapForm()
        return self.render_to_response(self.get_context_data(form=form))

    def post(self, request, *args, **kwargs):
        if self._target_map:
            form = TargetUriMapForm(request.POST, request.FILES, instance=self._target_map)
        else:
            form = TargetUriMapForm(request.POST, request.FILES)

        if form.is_valid():
            self._target_map = form.save()
            self._target_map.users.add(request.user)
            messages.add_message(request, messages.SUCCESS, "Target URI map saved")
            return redirect('target_map_list_view')
        else:
            messages.add_message(request, messages.ERROR, "Target URI map saving error")
        return self.render_to_response(self.get_context_data(form=form))

    @method_decorator(login_required)
    def dispatch(self, request, *args, **kwargs):
        if kwargs.get('map_id'):
            self._target_map = get_object_or_404(TargetUriMap, id=kwargs.get('map_id'))
            if request.user not in self._target_map.users.all():
                raise http.Http404
        self._user_target_maps = TargetUriMap.objects.filter(users__in=[request.user.id])
        return super(TargetMapView, self).dispatch(request, *args, **kwargs)


class TargetListView(ListView):
    template_name = "target_list.html"
    paginate_by = 50
    _user = None
    _crasher = None
    _abc_id = 0

    def get_context_data(self, **kwargs):
        context = super(TargetListView, self).get_context_data(**kwargs)
        context['crasher'] = self._crasher
        return context

    def get_queryset(self):
        filters = [Q(parent__isnull=True)]
        if not self._user.is_superuser:
            filters.append(Q(abc_id__in=self.user_abc_ids))
        if self._abc_id:
            filters.append(Q(abc_id=self._abc_id))
        qs = Target.objects.filter(reduce(operator.and_, filters)).order_by('-last_scan')
        return qs

    @method_decorator(login_required)
    def dispatch(self, request, *args, **kwargs):
        self._user = request.user
        if request.user.is_superuser:
            self._crasher = request.GET.get('crasher', "int")
        self._abc_id = request.GET.get('abc_id', 0)
        self.user_abc_ids = [item[0] for item in get_abc_scopes(request.user.username)]
        return super(TargetListView, self).dispatch(request, *args, **kwargs)


class TargetEditView(TemplateView):
    template_name = "target.html"
    paginate_by = 30
    _target = None

    def get_context_data(self, **kwargs):
        context = super(TargetEditView, self).get_context_data(**kwargs)
        context['target_info'] = self._target
        latest_old_scan_time = datetime.datetime.now() + datetime.timedelta(days=-90)
        filters = Q(target=self._target)
        filters.add(Q(target__parent=self._target), Q.OR)
        filters.add(Q(start__gte=latest_old_scan_time), Q.AND)
        context['target_scans'] = Scan.objects.select_related('scan_task').filter(filters).order_by('-id')[:30]
        return context

    def get(self, request, *args, **kwargs):
        if self._target.parent:
            return redirect('target_details_view', self._target.parent.id)
        form = TargetEditForm(self._target, request.user)
        return self.render_to_response(self.get_context_data(form=form))

    def post(self, request, *args, **kwargs):
        form = TargetEditForm(self._target, request.user, request.POST, request.FILES)
        if form.is_valid():
            form.save()
            messages.add_message(request, messages.SUCCESS, "Target details saved")
            return redirect('target_details_view', self._target.id)
        else:
            messages.add_message(request, messages.ERROR, "Target details saving error")
        return self.render_to_response(self.get_context_data(form=form))

    @method_decorator(login_required)
    def dispatch(self, request, *args, **kwargs):
        if kwargs.get('target_id'):
            users = []
            self._target = get_object_or_404(Target, id=kwargs.get('target_id'))
            if not request.user.is_superuser:
                if self._target and self._target.abc_id:
                    users = get_abc_members(self._target.abc_id)
                if not users or request.user.username not in users:
                    if request.user not in self._target.users.all():
                        raise PermissionDenied
        return super(TargetEditView, self).dispatch(request, *args, **kwargs)


class CrasherDashboardView(ListView):
    template_name = "crasher.html"
    paginate_by = 50
    _user = None
    _severity = VulnerabilityType.SEVERITY_MEDIUM
    _current_vuln_type = 0

    def get_context_data(self, **kwargs):
        context = super(CrasherDashboardView, self).get_context_data(**kwargs)
        context['vuln_types'] = VulnerabilityType.objects.filter(archived=False, severity__gte=self._severity)
        if self._current_vuln_type:
            context['vuln_type'] = self._current_vuln_type.id
        context['crasher'] = self._filter
        context['limit'] = self._limit
        if self._current_vuln_type:
            context['severity'] = self._current_vuln_type.severity
        else:
            context['severity'] = self._severity
        #XXX
        try:
            context['crasher_status'] = CrasherStatus.objects.all()[0]
        except Exception:
            pass
        context['crasher_form'] = CrasherStatusForm()
        return context

    def get_queryset(self):
        filters = [Q(false_positive__isnull=True)]
        filters.append(Q(vuln_type__is_internal=False))
        if self._current_vuln_type:
            filters.append(Q(vuln_type=self._current_vuln_type))
        else:
            if self._severity != VulnerabilityType.SEVERITY_IGNORE:
                filters.append(Q(vuln_type__severity__gte=self._severity))

        if self._filter == 'prod':
            filters.append(Q(scan__is_prod=True))
        if self._filter == 'int':
            filters.append(Q(scan__is_prod=False))

        qs = Vulnerability.objects.filter(reduce(operator.and_, filters)).order_by('-pk')[:self._limit]
        if not qs:
            return []
        return qs

    @method_decorator(login_required)
    @method_decorator(user_passes_test(lambda u: u.is_superuser))
    def dispatch(self, request, *args, **kwargs):
        self._user = request.user
        self._filter = request.GET.get('crasher', 'prod')
        self._severity = VulnerabilityType.SEVERITY_IGNORE
        if self._filter not in ['int', 'all', 'prod']:
            self._filter = 'all'
        try:
            self._severity = int(request.GET.get('severity', VulnerabilityType.SEVERITY_MEDIUM))
        except ValueError:
            raise http.Http404
        if self._severity not in list(VulnerabilityType.SEVERITIES.keys()):
            raise http.Http404
        self._current_vuln_type = None
        if request.GET.get('vuln_type'):
            self._current_vuln_type = get_object_or_404(VulnerabilityType, id=request.GET.get('vuln_type'))
        self._limit = int(request.GET.get('limit', 200))
        return super(CrasherDashboardView, self).dispatch(request, *args, **kwargs)

    def post(self, request, *args, **kwargs):
        form = CrasherStatusForm(request.POST)
        resp = {'status': 'fail'}
        if form.is_valid():
            crasher = get_object_or_404(CrasherStatus, id=1)
            if form.cleaned_data.get('run', False):
                crasher.status = CrasherStatus.STATUS_NEEDRUN
            else:
                crasher.status = CrasherStatus.STATUS_IDLE
            crasher.user = request.user
            crasher.save()
            resp = {'status': 'ok'}
        return HttpResponse(json.dumps(resp), content_type='application/json; charset=utf-8')


class IndexView(TemplateView):
    template_name = "index.html"
    _target = None

    def get_context_data(self, **kwargs):
        context = super(IndexView, self).get_context_data(**kwargs)
        return context

    def get(self, request, *args, **kwargs):
        form = NewScanForm(request.user)
        return self.render_to_response(self.get_context_data(form=form))

    def post(self, request, *args, **kwargs):
        form = NewScanForm(request.user, request.POST)
        target = None
        if form.is_valid():
            target_uri = form.cleaned_data.get('url')
            auth_profile_uid = form.cleaned_data.get('auth_profile_uid')
            scan_qs = Scan.objects.filter(slug=slugify_target_url(target_uri)).order_by('-id')[:1]
            if scan_qs:
                target = scan_qs[0].target
            if not target:
                target, new = Target.objects.get_or_create(name=target_uri, url=target_uri)
            if target.parent:
                target = target.parent
            if form.cleaned_data.get('abc_id') and not target.abc_id:
                target.abc_id = form.cleaned_data.get('abc_id')
                target.save()
            scan_profile = get_object_or_404(ScanProfile, id=form.cleaned_data.get('scan_profile_id'))
            task_name = 'Scan %s with profile "%s"' % (target.name, scan_profile.name)
            scan_task = ScanTask.objects.create(target=target, name=task_name, is_prod=False)
            ProfilesTasks.objects.get_or_create(scan_profile=scan_profile, scan_task=scan_task)
            auth_profile = None
            try:
                auth_profile = AuthProfile.objects.get(uid=auth_profile_uid)
            except AuthProfile.DoesNotExist:
                pass
            latest_old_scan_time = datetime.datetime.now() + datetime.timedelta(minutes=-settings.API_RATELIMIT_MINUTES)
            ongoing_scans = Scan.objects.filter(url=target_uri, status=Scan.ST_INPROGRESS,
                                                start__gte=latest_old_scan_time)[:1]
            if ongoing_scans:
                scan = ongoing_scans[0]
            else:
                scan = scan_task.create_scan(user=request.user, url=target_uri,
                                             auth_profile=auth_profile, send_mail=True)
                if scan:
                    scan_changed = False
                    if form.cleaned_data.get('report_ticket'):
                        scan.report_ticket = form.cleaned_data.get('report_ticket', '')
                        scan_changed = True
                    if form.cleaned_data.get('scanner_type') and request.user.is_superuser:
                        scan.scanner_type = form.cleaned_data.get('scanner_type')
                        scan_changed = True
                    if form.cleaned_data.get('collaborator_type') and request.user.is_superuser:
                        scan.collaborator_type = form.cleaned_data.get('collaborator_type')
                        scan_changed = True
                    if form.cleaned_data.get('rps'):
                        scan.throttle = 60000 / form.cleaned_data.get('rps')
                        scan_changed = True
                    if scan_changed:
                        scan.save()
                    run_scan.delay(scan.id)
                    messages.add_message(request, messages.SUCCESS, "Scan started successfully")
                else:
                    messages.add_message(request, messages.ERROR, "Scan starting error")
            return HttpResponseRedirect(reverse('show_report', kwargs={'scan_uid': scan.uid}) + '?progressive=True')
        else:
            messages.add_message(request, messages.ERROR, "Scan starting error")
        return self.render_to_response(self.get_context_data(form=form))

    @method_decorator(login_required)
    def dispatch(self, request, *args, **kwargs):
        return super(IndexView, self).dispatch(request, *args, **kwargs)


class MergeTargetView(TemplateView):
    template_name = "target_merge.html"
    _user = None

    def get(self, request, *args, **kwargs):
        form = TargetMergeForm()
        return self.render_to_response(self.get_context_data(form=form))

    def post(self, request, *args, **kwargs):
        form = TargetMergeForm(request.POST)
        if form.is_valid():
            t1 = get_object_or_404(Target, id=form.cleaned_data.get('target_id'))
            t2 = get_object_or_404(Target, id=form.cleaned_data.get('merge_id'))
            if t1.id != t2.id:
                if not t1.abc_id:
                    t1.abc_id = t2.abc_id
                for c in t2.children.all():
                    c.parent = t1
                    c.save()
                t2.parent = t1
                t2.save()
            for child in t1.children.all():
                for u in child.users.all():
                    t1.users.add(u)
            messages.add_message(request, messages.SUCCESS, "Merged")
        else:
            messages.add_message(request, messages.ERROR, "Merge error")
        return HttpResponseRedirect(reverse('target_merge_view'))


    @method_decorator(login_required)
    @method_decorator(user_passes_test(lambda u: u.is_superuser))
    def dispatch(self, request, *args, **kwargs):
        self._user = request.user
        return super(MergeTargetView, self).dispatch(request, *args, **kwargs)


class AntirobotWizardView(TemplateView):
    template_name = "antirobot_wizard.html"

    def get(self, request, *args, **kwargs):
        form = AntirobotWizardForm()
        return self.render_to_response(self.get_context_data(form=form))

    def post(self, request, *args, **kwargs):
        form = AntirobotWizardForm(request.POST)
        if form.is_valid():
            target = get_object_or_404(Target, id=form.cleaned_data.get('target_id'))
            resp = dict()

            if target.parent:
                target = target.parent

            if not target.antirobot_uid:
                target.antirobot_uid = 'molly-%s' % slugify(target.name)
                target.save()

            resp['uid'] = target.antirobot_uid
            parsed_host = urllib.parse.urlparse(target.url)
            resp['domain'] = parsed_host.netloc
            resp['config'] = render_to_string('collector.conf.tpl', resp)
            return HttpResponse(json.dumps(resp), content_type='application/json; charset=utf-8')

        return self.render_to_response(self.get_context_data(form=form))

    @method_decorator(login_required)
    def dispatch(self, request, *args, **kwargs):
        return super(AntirobotWizardView, self).dispatch(request, *args, **kwargs)


class StQueueSuggest(View):
    def get(self, request, *args, **kwargs):
        res = []
        # XXX: use robot credentials, not cookies
        tokens = {'Session_id': request.COOKIES.get('Session_id', ''),
                  'sessionid2': request.COOKIES.get('sessionid2', '')}
        if request.GET.getlist('persons'):
            for person in request.GET.getlist('persons'):
                res += get_st_queues_from_abc_by_username(person, tokens)
        # TODO: use abc not golem
        if request.GET.get('uri'):
            admins = get_admins_from_golem_by_uri(request.GET.get('uri'))
            for person in admins:
                res += get_st_queues_from_abc_by_username(person, tokens)
        if request.GET.get('target_id'):
            target = get_object_or_404(Target, id=request.GET.get('target_id'))
            if target.st_queue:
                res.append(target.st_queue)
            for person in target.users.all():
                res += get_st_queues_from_abc_by_username(person.username, tokens)
        if request.GET.getlist('vuln_id'):
            vuln = get_object_or_404(Vulnerability, id=request.GET.get('vuln_id'))
            target = vuln.scan.target
            if target.st_queue:
                res.append(target.st_queue)
            if target.abc_id:
                res += get_st_queue_by_abc_id(target.abc_id)
            for person in target.users.all():
                res += get_st_queues_from_abc_by_username(person.username, tokens)
        return HttpResponse(json.dumps(list(set(res))), content_type='application/json; charset=utf-8')


class ScanTargetEditView(TemplateView):
    template_name = "scan_target_edit.html"

    def get(self, request, *args, **kwargs):
        form = ScanTargetEditForm(self.scan, request.user)
        return self.render_to_response(self.get_context_data(scan=self.scan, form=form))

    def post(self, request, *args, **kwargs):
        form = ScanTargetEditForm(self.scan, request.user, request.POST)
        if form.is_valid():
            form.save()
            messages.add_message(request, messages.SUCCESS, "Scan updated")
        return self.render_to_response(self.get_context_data(scan=self.scan, form=form))

    @method_decorator(login_required)
    def dispatch(self, request, *args, **kwargs):
        self.scan = get_object_or_404(Scan, uid=kwargs.get('scan_uid'))
        return super(ScanTargetEditView, self).dispatch(request, *args, **kwargs)


class NewTargetView(TemplateView):
    template_name = "new_target.html"

    def get(self, request, *args, **kwargs):
        form = NewTargetForm(request.user)
        return self.render_to_response(self.get_context_data(form=form))

    def post(self, request, *args, **kwargs):
        form = NewTargetForm(request.user, request.POST)
        if form.is_valid():
            target = form.save()
            messages.add_message(request, messages.SUCCESS, "Target created")
            return HttpResponseRedirect(reverse('target_details_view', kwargs={'target_id': target.id}))
        messages.add_message(request, messages.ERROR, "Something goes wrong")
        return self.render_to_response(self.get_context_data(form=form))

    @method_decorator(login_required)
    def dispatch(self, request, *args, **kwargs):
        return super(NewTargetView, self).dispatch(request, *args, **kwargs)


class MDSProxyView(View):
    def get(self, request, *args, **kwargs):
        key = os.path.join(kwargs.get('prefix'), kwargs.get('postfix'))
        data = read_from_mds(key)
        res = HttpResponse(data)
        res['Content-Type-Options'] = 'nosniff'
        res['Content-Disposition'] = 'attachment; filename=raw.log'
        return HttpResponse(data)

    # XXX: we download request samples from agents
    def dispatch(self, request, *args, **kwargs):
        return super(MDSProxyView, self).dispatch(request, *args, **kwargs)
