import argparse
import datetime
import ipaddress
import json
import socket
import time

import requests


ip2mac = {}


def load_l23():
    with open("l23.txt", 'rb') as l23:
        return json.load(l23)


def update_l23(force=False):
    url = 'https://racktables.yandex.net/export/netmap/L23'
    now = time.time()

    try:
        with open("l23.txt", 'rb') as l23:
            l23 = json.load(l23)
            now = time.time()
            if (l23.get('timestamp') - now < 3600) and not force:
                return
    except (FileNotFoundError, ValueError):
        pass

    content = requests.get(url).text
    ip2mac = {}
    for record in content.strip().split('\n'):
        ip, mac = record.split()
        ip2mac[ip] = mac
    ip2mac['timestamp'] = now
    with open("l23.txt", 'wb') as l23:
        ip2mac = json.dumps(ip2mac)
        l23.write(ip2mac.encode())


def make_debby_record(entry, ports, comment=''):
    date = datetime.datetime.now().date()
    if '/' not in entry:
        phyaddr = ip2mac.get(entry, 'NO_MAC')
    else:
        phyaddr = 'NET'
    result = []
    for port in ports.split(','):
        proto, port = port.split('/')
        result.append(f'{date},{phyaddr},{entry},{proto},{port},{comment}')
    return '\n'.join(result)


def make_watcher_record(entry, ports, comment=''):
    result = {}
    if '/' not in entry:
        result['mac'] = ip2mac.get(entry, 'NO_MAC')
        result['host'] = entry
    else:
        result['mac'] = 'NET'
        result['net'] = entry
    if comment:
        result['comment'] = comment
    result['ports'] = ports.split(',')
    return json.dumps(result)


def main():
    global ip2mac
    parser = argparse.ArgumentParser(description='Create net whitelist records.')
    parser.add_argument('-m', '--mode', action='store', default='debby',
                        choices=['debby', 'watcher'],
                        help='records format (default: debby)')
    parser.add_argument('-e', '--entry', action='append', required=True)
    parser.add_argument('-p', '--ports', action='append', required=True)
    parser.add_argument('-c', '--comment', action='store', default='')
    parser.add_argument('-n', '--no-update', help='skip l23.txt update', action='store_true')
    parser.add_argument('-u', '--update', help='force l23.txt update', action='store_true')
    parser.add_argument('-o', '--output', help='file to append output', action='store')
    args = parser.parse_args()

    if len(args.entry) != len(args.ports):
        print('Hosts/networks and ports don\'t match!')
        exit(1)
    if not args.no_update:
        update_l23(args.update)

    ip2mac = load_l23()

    if args.mode == 'debby':
        make_record = make_debby_record
    elif args.mode == 'watcher':
        make_record = make_watcher_record

    result = []
    for i in range(len(args.entry)):
        entry = args.entry[i]
        ports = args.ports[i]
        try:
            if '/' not in entry:
                ipaddress.ip_address(entry)
        except ValueError:
            entry = {data[4][0] for data in socket.getaddrinfo(entry, 0)}
        if type(entry) is set:
            for ip in entry:
                record = make_record(ip, ports, args.comment)
                result.append(record)
        else:
            record = make_record(entry, ports, args.comment)
            result.append(record)

    result = '\n'.join(result)
    if not args.output:
        print(result)
    else:
        with open(args.output, 'ab') as of:
            of.write(result.encode())
            of.write(b'\n')


if __name__ == '__main__':
    main()
