package gosecure

import (
	"io/ioutil"
	"os"
	"path/filepath"
	"regexp"
	"strconv"
	"strings"

	"a.yandex-team.ru/library/go/core/xerrors"
)

const (
	apparmor           = "apparmor"
	profiles           = "profiles"
	apparmorModulePath = "/sys/module/apparmor"
	profilesLineRegex  = `^([^\(]+)\s+\((\w+)\)$`
	regexMatchLen      = 3
	procDir            = "/proc"
)

var (
	profilesLine *regexp.Regexp
)

type AppArmorMode string

const (
	AppArmorEnforce        AppArmorMode = "enforce"
	AppArmorComplain       AppArmorMode = "complain"
	AppArmorUnconfinedMode AppArmorMode = "unconfined"
	AppArmorKill           AppArmorMode = "kill"
	AppArmorMixed          AppArmorMode = "mixed"
)

type AppArmorProfileName string

type AppArmorProfilesStatus map[AppArmorProfileName]AppArmorMode

type AppArmorProcessStatus struct {
	Profile AppArmorProfileName
	Mode    AppArmorMode
	Pid     uint64
}

type AppArmorStatus struct {
	ModuleLoaded      bool
	Enabled           bool
	Profiles          AppArmorProfilesStatus
	SecurityFsPath    string
	ConfinedProcesses []AppArmorProcessStatus
}

func init() {
	profilesLine = regexp.MustCompile(profilesLineRegex)
}

func isAppArmorModuleLoaded() bool {
	_, err := os.Stat(apparmorModulePath)
	return !os.IsNotExist(err)
}

func isAppArmorModuleEnabled() bool {
	data, err := ioutil.ReadFile(filepath.Join(apparmorModulePath, "parameters", "enabled"))
	if err == nil {
		return strings.Trim(string(data), "\n") == "Y"
	}

	return false
}

func readLoadedProfiles(securityFsPath string) (AppArmorProfilesStatus, error) {
	profilesFile := filepath.Join(securityFsPath, apparmor, profiles)
	if _, err := os.Stat(profilesFile); os.IsNotExist(err) {
		return nil, err
	}

	data, err := ioutil.ReadFile(profilesFile)
	if err != nil {
		return nil, err
	}

	status := AppArmorProfilesStatus{}
	lines := strings.Split(string(data), "\n")
	for num := range lines {
		match := profilesLine.FindStringSubmatch(lines[num])
		if len(match) == regexMatchLen {
			status[AppArmorProfileName(match[1])] = AppArmorMode(match[2])
		}
	}

	return status, nil
}

func readProcessAttr(procPath string) (string, string, error) {
	data, err := ioutil.ReadFile(procPath)
	if err != nil {
		return "", "", nil
	}

	trm := strings.Trim(string(data), "\n")
	if match := profilesLine.FindStringSubmatch(trm); len(match) == regexMatchLen {
		return match[1], match[2], nil
	}

	return "", "", xerrors.New("couldn't find apparmor profile")
}

func readProcessesStatuses() ([]AppArmorProcessStatus, error) {
	procList, err := ioutil.ReadDir(procDir)
	if err != nil {
		return nil, err
	}

	status := make([]AppArmorProcessStatus, 0)
	for num := range procList {
		if isDigit(procList[num].Name()) {
			profilePath := filepath.Join(procDir, procList[num].Name(), "attr", "current")
			_, err := os.Stat(profilePath)

			if err != nil {
				continue
			}

			if profile, mode, err := readProcessAttr(profilePath); err != nil {
				continue
			} else {
				pid, _ := strconv.ParseUint(procList[num].Name(), 10, 64)
				status = append(status, AppArmorProcessStatus{
					AppArmorProfileName(profile),
					AppArmorMode(mode),
					pid,
				})
			}
		}
	}

	return status, nil
}

func GetAppArmorStatus() (*AppArmorStatus, error) {
	securityFsPath, err := FindSecurityFs()
	if err != nil {
		return nil, err
	}

	profiles, err := readLoadedProfiles(securityFsPath)
	if err != nil {
		return nil, err
	}

	procStatuses, err := readProcessesStatuses()
	if err != nil {
		return nil, err
	}

	return &AppArmorStatus{
		isAppArmorModuleLoaded(),
		isAppArmorModuleEnabled(),
		profiles,
		securityFsPath,
		procStatuses,
	}, nil
}

func GetAppArmorProfiles() (AppArmorProfilesStatus, error) {
	securityFsPath, err := FindSecurityFs()
	if err != nil {
		return nil, err
	}

	return readLoadedProfiles(securityFsPath)
}

func GetProcessesConfinementStatus() ([]AppArmorProcessStatus, error) {
	return readProcessesStatuses()
}
