package apt

import (
	"io/ioutil"
	"os"
	"path"
	"strings"

	"golang.org/x/crypto/openpgp"
	"golang.org/x/crypto/openpgp/packet"

	"a.yandex-team.ru/library/go/core/xerrors"
)

const (
	aptKeyringFile = "/etc/apt/trusted.gpg"
	aptKeyringDir  = "/etc/apt/trusted.gpg.d/"
)

type Key struct {
	Entity   *openpgp.Entity
	Filename string
}

type KeyRing []*Key

type ErrorCollection []error

func (e ErrorCollection) FormatErrors() error {
	var sb strings.Builder
	for _, err := range e {
		sb.WriteString(err.Error())
		sb.WriteString(";")
	}

	return xerrors.New(sb.String())
}

func readKeychainFromFile(filename string) (openpgp.EntityList, error) {
	file, err := os.Open(filename)
	if err != nil {
		return nil, err
	}
	defer file.Close()

	return openpgp.ReadKeyRing(file)
}

func IdentityToSting(i map[string]*openpgp.Identity) string {
	if len(i) == 1 {
		for k := range i {
			return k
		}
	}

	var sb strings.Builder
	for key := range i {
		sb.WriteString(key)
		sb.WriteString(";")
	}
	return sb.String()
}

func NewAptKeyRing() (KeyRing, ErrorCollection) {
	files, err := ioutil.ReadDir(aptKeyringDir)
	if err != nil {
		return nil, ErrorCollection{err}
	}

	keychains := make([]string, 0, len(files)+1)
	for item := range files {
		if !files[item].IsDir() {
			keychains = append(keychains, path.Join(aptKeyringDir, files[item].Name()))
		}
	}

	keychains = append(keychains, aptKeyringFile)
	errs := make(ErrorCollection, 0)
	result := make(KeyRing, 0)

	for _, chain := range keychains {
		entities, err := readKeychainFromFile(chain)
		if err != nil {
			errs = append(errs, err)
			continue
		}

		for _, entity := range entities {
			result = append(result, &Key{entity, chain})
		}
	}

	return result, errs
}

func AlgoToString(algo packet.PublicKeyAlgorithm) string {
	switch algo {
	case packet.PubKeyAlgoRSA:
		return "rsa"
	case packet.PubKeyAlgoRSAEncryptOnly:
		return "rsa_encrypt"
	case packet.PubKeyAlgoRSASignOnly:
		return "rsa_sign"
	case packet.PubKeyAlgoElGamal:
		return "elgamal"
	case packet.PubKeyAlgoDSA:
		return "dsa"
	case packet.PubKeyAlgoECDH:
		return "ecdh"
	case packet.PubKeyAlgoECDSA:
		return "ecdsa"
	default:
		return "unknown"
	}
}
