package dbus

import (
	"fmt"
	"log"
	"strings"
	"time"

	godbus "github.com/godbus/dbus/v5"

	"a.yandex-team.ru/library/go/core/xerrors"
)

const (
	SignalUserNew            = "UserNew"
	SignalUserRemoved        = "UserRemoved"
	SignalSessionNew         = "SessionNew"
	SignalSessionRemoved     = "SessionRemoved"
	SignalSeatNew            = "SeatNew"
	SignalSeatRemoved        = "SeatRemoved"
	SignalPrepareForShutdown = "PrepareForShutdown"
	SignalPrepareForSleep    = "PrepareForSleep"
)

var subscription = []string{
	SignalUserNew,
	SignalUserRemoved,
	SignalSessionNew,
	SignalSessionRemoved,
	SignalSeatNew,
	SignalSeatRemoved,
	SignalPrepareForShutdown,
	SignalPrepareForSleep,
}

const (
	SignalUnitNew          = "UnitNew"
	SignalUnitRemoved      = "UnitRemoved"
	SignalJobNew           = "JobNew"
	SignalJobRemoved       = "JobRemoved"
	SignalStartupFinished  = "StartupFinished"
	SignalUnitFilesChanged = "UnitFilesChanged"
	SignalReloading        = "Reloading"
	channelCap             = 1024
)

var systemdSub = []string{
	SignalUnitNew,
	SignalUnitRemoved,
	SignalJobNew,
	SignalJobRemoved,
	SignalStartupFinished,
	SignalUnitFilesChanged,
	SignalReloading,
}

type ListenerType int

type EventRecord struct {
	Timestamp   int64
	EventRecord string
}

type listener struct {
	channel            chan *godbus.Signal
	connection         *godbus.Conn
	activeSubscription bool
	shutdownChannel    chan bool
	buffer             *Buffer
	handlers           []SignalHandler
}

type SignalHandler = func(s *godbus.Signal)

func NewListener(handlers ...SignalHandler) (*listener, error) {
	conn, err := godbus.SystemBus()
	if err != nil {
		return nil, err
	}

	lst := &listener{
		nil,
		conn,
		false,
		nil,
		nil,
		handlers,
	}

	return lst, nil
}

func (l *listener) Start(buffer *Buffer) error {
	if l.activeSubscription {
		return xerrors.New("second Start() call, nothing will be done")

	}
	l.shutdownChannel = make(chan bool)
	l.channel = make(chan *godbus.Signal)
	l.buffer = buffer

	for _, sub := range systemdSub {
		l.connection.BusObject().Call(
			"org.freedesktop.DBus.AddMatch",
			0,
			fmt.Sprintf("type='signal',interface='org.freedesktop.systemd1.Manager',member='%s'", sub))
	}

	for _, sub := range subscription {
		l.connection.BusObject().Call(
			"org.freedesktop.DBus.AddMatch",
			0,
			fmt.Sprintf("type='signal',interface='org.freedesktop.login1.Manager',member='%s'", sub))
	}

	l.connection.Signal(l.channel)

	for {
		select {
		case msg := <-l.channel:
			l.buffer.Append(msg)
			go func() {
				for _, v := range l.handlers {
					v(msg)
				}
			}()
		case <-l.shutdownChannel:
			return nil
		}
	}
}

func (l *listener) Shutdown() {
	select {
	case l.shutdownChannel <- true:
		time.Sleep(2 * time.Second)
	case <-time.After(3 * time.Second):
		log.Println("shutdown timeout, closing all channels")
	}

	close(l.channel)
	l.channel = nil
	close(l.shutdownChannel)
	l.shutdownChannel = nil
}

// https://www.freedesktop.org/wiki/Software/systemd/dbus/
func EventToRecord(event *DbusEvent) *EventRecord {
	if event == nil {
		return nil
	}

	parts := strings.Split(string(event.Signal.Path), ".")
	name := parts[len(parts)-1]

	switch name {
	case SignalPrepareForSleep, SignalPrepareForShutdown:
		return &EventRecord{event.Timestamp, fmt.Sprintf("event=%s value=%t", event.Signal.Path, event.Signal.Body[0].(bool))}
	case SignalUserNew, SignalUserRemoved:
		return &EventRecord{event.Timestamp, fmt.Sprintf("event=%s uid=%d", event.Signal.Path, event.Signal.Body[0].(int64))}
	case SignalSessionNew, SignalSessionRemoved:
		return &EventRecord{event.Timestamp, fmt.Sprintf("event=%s sessionid=%d", event.Signal.Path, event.Signal.Body[0].(int64))}
	case SignalSeatNew, SignalSeatRemoved:
		return &EventRecord{event.Timestamp, fmt.Sprintf("event=%s seatid=%d", event.Signal.Path, event.Signal.Body[0].(int64))}
	case SignalUnitNew, SignalUnitRemoved:
		return &EventRecord{event.Timestamp, fmt.Sprintf("event=%s unit=%s", event.Signal.Path, event.Signal.Body[0].(string))}
	case SignalJobNew:
		return &EventRecord{event.Timestamp, fmt.Sprintf("event=%s jobid=%d unit=%s", event.Signal.Path, event.Signal.Body[0].(int64), event.Signal.Body[2].(string))}
	case SignalJobRemoved:
		return &EventRecord{event.Timestamp, fmt.Sprintf("event=%s jobid=%d unit=%s result=%s", event.Signal.Path, event.Signal.Body[0].(int64), event.Signal.Body[2].(string), event.Signal.Body[3].(string))}
	case SignalReloading:
		return &EventRecord{event.Timestamp, fmt.Sprintf("event=%s starting=%t", event.Signal.Path, event.Signal.Body[0].(bool))}
	case SignalUnitFilesChanged:
		return &EventRecord{event.Timestamp, fmt.Sprintf("event=%s", event.Signal.Path)}
	}

	return &EventRecord{event.Timestamp, fmt.Sprintf("event=%s value=%v", event.Signal.Path, event.Signal.Body)}
}
