package logind

import (
	"encoding/xml"
	"fmt"
	"reflect"
	"strings"

	"github.com/godbus/dbus/v5"
)

const (
	logindDest       = "org.freedesktop.login1"
	sessionPath      = "/org/freedesktop/login1/session"
	methodIntrospect = "org.freedesktop.DBus.Introspectable.Introspect"
)

type User struct {
	UserID uint32
	Path   dbus.ObjectPath
}

type Seat struct {
	SeatID string
	Path   dbus.ObjectPath
}

type Session struct {
	ID                     string
	User                   User
	Name                   string
	Timestamp              uint64
	TimestampMonotonic     uint64
	VTNr                   uint32
	Seat                   Seat
	TTY                    string
	Display                string
	Remote                 bool
	RemoteHost             string
	RemoteUser             string
	Service                string
	Scope                  string
	Leader                 uint32
	Audit                  uint32
	Type                   string
	Class                  string
	Active                 bool
	State                  string
	IdleHint               bool
	IdleSinceHint          uint64
	IdleSinceHintMonotonic uint64
}

func GetAllSessions() ([]dbus.ObjectPath, error) {
	conn, err := dbus.SystemBus()
	if err != nil {
		return nil, err
	}

	var introspection interface{}
	if err := conn.Object(logindDest, sessionPath).Call(methodIntrospect, 0).Store(&introspection); err != nil {
		return nil, err
	}

	i := introspection.(string)

	parser := xml.NewDecoder(strings.NewReader(i))
	attrValues := make([]string, 0)
	for {
		token, err := parser.Token()
		if err != nil {
			break
		}

		switch t := token.(type) {
		case xml.StartElement:
			element := xml.StartElement(t)
			if element.Name.Local == "node" && len(element.Attr) != 0 && element.Attr[0].Name.Local == "name" {
				if element.Attr[0].Value != "self" {
					attrValues = append(attrValues, element.Attr[0].Value)
				}
			}
		}
	}

	result := make([]dbus.ObjectPath, len(attrValues))
	for index, attr := range attrValues {
		result[index] = dbus.ObjectPath(fmt.Sprintf(sessionPath+"/%s", attr))
	}
	return result, nil
}

func GetSessionInfo(path dbus.ObjectPath) (*Session, error) {
	conn, err := dbus.SystemBus()
	if err != nil {
		return nil, err
	}

	session := Session{}
	for i := 0; i < reflect.ValueOf(session).NumField(); i++ {
		name := reflect.ValueOf(session).Type().Field(i).Name
		propValue, err := conn.Object(logindDest, path).GetProperty(fmt.Sprintf("org.freedesktop.login1.Session.%s", name))
		if err != nil {
			return nil, err
		}

		if name == "User" {
			session.User = User{
				propValue.Value().([]interface{})[0].(uint32),
				propValue.Value().([]interface{})[1].(dbus.ObjectPath),
			}
		} else if name == "Seat" {
			session.Seat = Seat{
				propValue.Value().([]interface{})[0].(string),
				propValue.Value().([]interface{})[1].(dbus.ObjectPath),
			}
		} else {
			reflect.ValueOf(&session).Elem().FieldByName(name).Set(reflect.ValueOf(propValue.Value()))
		}
	}

	return &session, nil
}

func GetAllSessionObjects() ([]Session, error) {
	objects, err := GetAllSessions()
	if err != nil {
		return nil, err
	}

	result := make([]Session, len(objects))
	for index, obj := range objects {
		session, err := GetSessionInfo(obj)
		if err != nil {
			return nil, err
		}

		result[index] = *session
	}

	return result, nil
}
