package dbus

import (
	"fmt"

	"github.com/godbus/dbus/v5"

	"a.yandex-team.ru/library/go/core/xerrors"
)

type systemdListener struct {
	conn            *dbus.Conn
	channel         chan *dbus.Signal
	isInitialized   bool
	buffer          *Buffer
	handlers        []SignalHandler
	shutdownChannel chan bool
}

func NewSystemdListener(handlers ...SignalHandler) *systemdListener {
	return &systemdListener{
		handlers: handlers,
	}
}

func (l *systemdListener) Start(buffer *Buffer) error {
	if l.isInitialized {
		return xerrors.New("connection has already been initialized")
	}

	conn, err := dbus.SystemBus()
	if err != nil {
		return err
	}

	l.conn = conn
	for _, sub := range systemdSub {
		l.conn.BusObject().Call("org.freedesktop.DBus.AddMatch", 0, fmt.Sprintf("type='signal',interface='org.freedesktop.systemd1.Manager',member='%s'", sub))
	}

	ch := make(chan *dbus.Signal, channelCap)
	l.conn.Signal(ch)
	l.channel = ch
	l.buffer = buffer

	for {
		select {
		case msg := <-l.channel:
			l.buffer.Append(msg)
			go func() {
				for _, v := range l.handlers {
					v(msg)
				}
			}()
		case <-l.shutdownChannel:
			return nil
		}
	}
}
