package systemd

import (
	"fmt"

	"github.com/godbus/dbus/v5"

	"a.yandex-team.ru/library/go/core/xerrors"
)

const (
	systemdDest   = "org.freedesktop.systemd1"
	systemdPath   = "/org/freedesktop/systemd1"
	listUnits     = "org.freedesktop.systemd1.Manager.ListUnits"
	listUnitFiles = "org.freedesktop.systemd1.Manager.ListUnitFiles"
	listJobs      = "org.freedesktop.systemd1.Manager.ListJobs"
	unitPath      = "org.freedesktop.systemd1.Manager.UnitPath"
)

/*
Check out documentation on ListUnits method
https://www.freedesktop.org/wiki/Software/systemd/dbus/
*/
type UnitInfo struct {
	Name          string
	Description   string
	LoadState     string
	ActiveState   string
	SubState      string
	FollowedBy    string
	ObjectPath    dbus.ObjectPath
	JobID         uint32
	JobType       string
	JobObjectPath dbus.ObjectPath
}

func ListUnits() ([]UnitInfo, error) {
	conn, err := dbus.SystemBus()
	if err != nil {
		return nil, err
	}

	var units []UnitInfo
	if err := conn.Object(systemdDest, systemdPath).Call(listUnits, 0).Store(&units); err != nil {
		return nil, err
	}

	if units == nil {
		return nil, xerrors.New("nil result returned")
	}

	return units, nil
}

type UnitFileInfo struct {
	Path   string
	Status string
}

func ListUnitFiles() ([]UnitFileInfo, error) {
	conn, err := dbus.SystemBus()
	if err != nil {
		return nil, err
	}

	var files []interface{}
	if err := conn.Object(systemdDest, systemdPath).Call(listUnitFiles, 0).Store(&files); err != nil {
		return nil, err
	}

	fmt.Println(files)
	return nil, nil
}

type JobInfo struct {
	ID       uint32
	UnitName string
	Type     string
	State    string
	Object   dbus.ObjectPath
	Unit     dbus.ObjectPath
}

func ListJobs() ([]JobInfo, error) {
	conn, err := dbus.SystemBus()
	if err != nil {
		return nil, err
	}

	var jobs []JobInfo
	if err := conn.Object(systemdDest, systemdPath).Call(listJobs, 0).Store(&jobs); err != nil {
		return nil, err
	}

	return jobs, nil
}

func UnitPath() ([]string, error) {
	conn, err := dbus.SystemBus()
	if err != nil {
		return nil, err
	}

	paths, err := conn.Object(systemdDest, systemdPath).GetProperty(unitPath)
	if err != nil {
		return nil, err
	}

	return paths.Value().([]string), nil
}
