package gosecure

import (
	"io/ioutil"
	"strconv"
	"strings"
)

const (
	LSMpath             = "/sys/kernel/security/lsm"
	YamaPtraceScopeFile = "/proc/sys/kernel/yama/ptrace_scope"
)

func GetEnabledLSMs() ([]string, error) {
	data, err := ioutil.ReadFile(LSMpath)
	if err != nil {
		return nil, err
	}

	return strings.Split(string(data), ","), nil
}

/* For more details check out
https://www.kernel.org/doc/html/latest/admin-guide/LSM/Yama.html
0 - classic ptrace permissions:
    a process can PTRACE_ATTACH to any other process running under the same uid, as long as it is dumpable (i.e. did not transition uids, start privileged, or have called prctl(PR_SET_DUMPABLE...) already). Similarly, PTRACE_TRACEME is unchanged.
1 - restricted ptrace:
    a process must have a predefined relationship with the inferior it wants to call PTRACE_ATTACH on. By default, this relationship is that of only its descendants when the above classic criteria is also met. To change the relationship, an inferior can call prctl(PR_SET_PTRACER, debugger, ...) to declare an allowed debugger PID to call PTRACE_ATTACH on the inferior. Using PTRACE_TRACEME is unchanged.
2 - admin-only attach:
    only processes with CAP_SYS_PTRACE may use ptrace, either with PTRACE_ATTACH or through children calling PTRACE_TRACEME.
3 - no attach:
    no processes may use ptrace with PTRACE_ATTACH nor via PTRACE_TRACEME. Once set, this sysctl value cannot be changed.
*/

type YamaStatus int

const (
	YamaInvalidStatusValue          YamaStatus = -1
	YamaClassicPtracePermissions    YamaStatus = 0
	YamaRestrictedPtracePermissions YamaStatus = 1
	YamaAdminOnlyAttach             YamaStatus = 2
	YamaNoAttach                    YamaStatus = 3
)

func GetYAMAPtraceScope() (YamaStatus, error) {
	data, err := ioutil.ReadFile(YamaPtraceScopeFile)
	if err != nil {
		return YamaInvalidStatusValue, err
	}

	status, err := strconv.Atoi(string(data))
	if err != nil {
		return YamaInvalidStatusValue, err
	} else {
		return YamaStatus(status), nil
	}
}
