package gosecure

import (
	"fmt"
	"io/ioutil"
	"os"
	"path"
	"path/filepath"
	"strings"
)

// Some ideas were taken from
// https://github.com/lorenzosaino/go-sysctl/
const (
	sysctlBase = "/proc/sys"
)

func pathFromSysctlKey(key string) string {
	return path.Join(sysctlBase, strings.Replace(key, ".", "/", -1))
}

func readSysctl(key string) (string, error) {
	data, err := ioutil.ReadFile(pathFromSysctlKey(key))
	if err != nil {
		return "", err
	}

	return strings.TrimRight(string(data), "\n"), nil
}

func GetSysctlValue(key string) (string, error) {
	return readSysctl(key)
}

func keyFromPath(path string) string {
	subPath := strings.TrimPrefix(path, sysctlBase)
	return strings.Replace(subPath, "/", ".", -1)
}

func readFile(path string) (string, error) {
	data, err := ioutil.ReadFile(path)
	if err != nil {
		return "", err
	}
	return strings.TrimSpace(string(data)), nil
}

func AllSysctls() (map[string]string, error) {
	result := make(map[string]string)
	err := filepath.Walk(sysctlBase, func(path string, info os.FileInfo, err error) error {
		if err != nil {
			return fmt.Errorf("cannot read sysctl: %s", err.Error())
		}

		if info.IsDir() {
			return nil
		}

		key := keyFromPath(path)
		val, err := readFile(path)
		if err != nil {
			return fmt.Errorf("error reading %s: %s", path, err.Error())
		}

		result[key] = val
		return nil
	})

	if err != nil {
		return nil, err
	}

	return result, nil
}
