package main

import (
	"context"
	"encoding/hex"
	"flag"
	"fmt"
	"log"
	"os"
	"os/signal"
	"syscall"

	"github.com/kolide/osquery-go"
	"github.com/kolide/osquery-go/plugin/table"

	"a.yandex-team.ru/security/osquery/extensions/gosecure/apt"
)

func AptKeysTable() []table.ColumnDefinition {
	return []table.ColumnDefinition{
		table.TextColumn("key_id_short"),
		table.TextColumn("key_id"),
		table.TextColumn("identity_name"),
		table.TextColumn("fingerprint"),
		table.TextColumn("algo"),
		table.TextColumn("is_subkey"),
		table.TextColumn("creation_time"),
		table.TextColumn("filename"),
	}
}

func GenerateAptKeyTableData(ctx context.Context, queryCtx table.QueryContext) ([]map[string]string, error) {
	keys, errors := apt.NewAptKeyRing()
	if len(errors) != 0 {
		return nil, errors.FormatErrors()
	}

	resultTable := make([]map[string]string, 0, len(keys))
	for _, key := range keys {
		resultTable = append(resultTable, map[string]string{
			"key_id_short":  key.Entity.PrimaryKey.KeyIdShortString(),
			"key_id":        key.Entity.PrimaryKey.KeyIdString(),
			"identity_name": apt.IdentityToSting(key.Entity.Identities),
			"fingerprint":   hex.EncodeToString(key.Entity.PrimaryKey.Fingerprint[:]),
			"algo":          apt.AlgoToString(key.Entity.PrimaryKey.PubKeyAlgo),
			"is_subkey":     fmt.Sprintf("%t", key.Entity.PrimaryKey.IsSubkey),
			"creation_time": key.Entity.PrimaryKey.CreationTime.String(),
			"filename":      key.Filename,
		})
	}

	return resultTable, nil
}

const (
	aptExtensionName = "apt_extension"
	aptKeysTableName = "apt_keys"
)

func main() {
	flag.Parse()
	var socketPath string
	if flag.NArg() != 1 {
		fmt.Printf("invalid path to osquery socket\n")
		os.Exit(1)
	} else {
		socketPath = flag.Arg(0)
	}

	if _, err := os.Stat(socketPath); os.IsNotExist(err) {
		fmt.Println("Can't find socket path", socketPath)
		os.Exit(1)
	}

	srv, err := osquery.NewExtensionManagerServer(aptExtensionName, socketPath)
	if err != nil {
		log.Fatalf("error calling extension %s\n", aptExtensionName)
	}

	srv.RegisterPlugin(table.NewPlugin(aptKeysTableName, AptKeysTable(), GenerateAptKeyTableData))

	idleConnsClosed := make(chan struct{})
	go func() {
		gracefulStop := make(chan os.Signal, 1)

		signal.Notify(gracefulStop, syscall.SIGTERM)
		signal.Notify(gracefulStop, syscall.SIGINT)
		<-gracefulStop

		if err := srv.Shutdown(context.Background()); err != nil {
			log.Printf("Server shutdown error: %v", err)
		}
		close(idleConnsClosed)

	}()

	if err := srv.Run(); err != nil {
		log.Fatal(err)
	}
	<-idleConnsClosed
}
