package main

import (
	"context"
	"flag"
	"fmt"
	"log"
	"os"
	"os/signal"
	"syscall"

	"github.com/kolide/osquery-go"
	"github.com/kolide/osquery-go/plugin/table"

	"a.yandex-team.ru/security/osquery/extensions/gosecure/dbus"
)

const (
	bufferSize              = 16 * 1024
	DBusUserEventsTableName = "dbus_user_events"
	DBusExtensionName       = "dbus"
)

func DBusUserEventsTable() []table.ColumnDefinition {
	return []table.ColumnDefinition{
		table.BigIntColumn("timestamp"),
		table.TextColumn("event"),
	}
}

func GenerateDBusTableData(buffer *dbus.Buffer) ([]map[string]string, error) {
	data := buffer.GetAll()
	result := make([]map[string]string, len(data))
	for i := range data {
		event := dbus.EventToRecord(data[i])
		if event == nil {
			result[i] = map[string]string{
				"timestamp": fmt.Sprintf("%d", data[i].Timestamp),
				"event":     fmt.Sprintf("value=%v", data[i].Signal),
			}
		} else {
			result[i] = map[string]string{
				"timestamp": fmt.Sprintf("%d", data[i].Timestamp),
				"event":     event.EventRecord,
			}
		}
	}

	return result, nil
}

func main() {
	flag.Parse()
	var socketPath string

	if flag.NArg() != 1 {
		fmt.Println("invalid path to osquery socket")
		os.Exit(1)
	} else {
		socketPath = flag.Arg(0)
	}

	if _, err := os.Stat(socketPath); os.IsNotExist(err) {
		fmt.Println("can't find socket path", socketPath)
		os.Exit(1)
	}

	buffer := dbus.NewBuffer(bufferSize)
	listener, err := dbus.NewListener()
	if err != nil {
		fmt.Println("can't initialize a new dbus subscription:", err.Error())
		os.Exit(1)
	}

	server, err := osquery.NewExtensionManagerServer(DBusExtensionName, socketPath)
	if err != nil {
		fmt.Println("Error calling extension:", err.Error())
		os.Exit(1)
	}
	server.RegisterPlugin(table.NewPlugin(DBusUserEventsTableName, DBusUserEventsTable(), func(ctx context.Context, queryContext table.QueryContext) (strings []map[string]string, e error) {
		return GenerateDBusTableData(buffer)
	}))

	if err := listener.Start(buffer); err != nil {
		fmt.Println("can't start listening to dbus events:", err.Error())
		os.Exit(1)
	}

	idleConnsClosed := make(chan struct{})
	go func() {
		gracefulStop := make(chan os.Signal, 1)

		signal.Notify(gracefulStop, syscall.SIGTERM)
		signal.Notify(gracefulStop, syscall.SIGINT)
		<-gracefulStop

		defer listener.Shutdown()
		if err := server.Shutdown(context.Background()); err != nil {
			log.Printf("Server shutdown error: %v", err)
		}
		close(idleConnsClosed)

	}()

	if err := server.Run(); err != nil {
		log.Fatal(err)
	}
	<-idleConnsClosed
}
