package main

import (
	"encoding/json"
	"flag"
	"log"
	"net/http"
	_ "net/http/pprof"
	"os"
	"runtime"
	"strconv"

	"a.yandex-team.ru/security/osquery/extensions/osquery-fim/internal/fim"
	"a.yandex-team.ru/security/osquery/extensions/osquery-fim/internal/platform"
)

var (
	socketPath  = flag.String("socket", "", "path to the osquery socket")
	verboseFlag = flag.Bool("verbose", false, "enable additional logging")
	_           = flag.String("timeout", "", "ignored, required by osquery")
	_           = flag.String("interval", "", "ignored, required by osquery")

	configPath = flag.String("config", "", "path to the config file")

	// Debug options.
	standalone      = flag.Bool("standalone", false, "run as a CLI app, not as an osquery plugin")
	dumpPath        = flag.String("dump", "", "if running standalone, dump the hashes to this file")
	disablePeriodic = flag.Bool("disable-periodic", false, "disable periodic rehashing, useful for debug")
	pprofPort       = flag.Int("pprof-port", 0, "open pprof interface at this port")
)

func main() {
	log.SetPrefix("[osquery-fim] ")

	flag.Parse()

	finalConfigPath := *configPath
	if finalConfigPath == "" {
		finalConfigPath = platform.GetDefaultConfigPath()
	}

	if finalConfigPath == "" {
		flag.Usage()
		log.Fatalf("FATAL: config is required\n")
	}

	if *pprofPort != 0 {
		runtime.SetBlockProfileRate(100)
		runtime.SetMutexProfileFraction(100)
		go func() {
			log.Println(http.ListenAndServe("localhost:"+strconv.Itoa(*pprofPort), nil))
		}()
	}

	if !*standalone {
		if *socketPath == "" {
			log.Fatalf("FATAL: invalid path to osquery socket\n")
		}

		if *dumpPath != "" {
			log.Fatalf("FATAL: debug-dump not allowed when not running as standalone")
		}
	}

	f, err := os.Open(finalConfigPath)
	if err != nil {
		log.Fatalf("FATAL: could not open config %s: %v\n", *configPath, err)
	}
	d := json.NewDecoder(f)
	config := fim.Config{}
	err = d.Decode(&config)
	if err != nil {
		log.Fatalf("FATAL: could not parse config %s: %v\n", *configPath, err)
	}

	verbose := false
	if *verboseFlag {
		verbose = true
	} else if config.Verbose != nil {
		verbose = *config.Verbose
	}
	verboseNotify := false
	if config.VerboseNotify != nil {
		verboseNotify = *config.VerboseNotify
	}
	debugConfig := fim.Debug{
		DumpPath:        *dumpPath,
		Verbose:         verbose,
		VerboseNotify:   verboseNotify,
		DisablePeriodic: *disablePeriodic,
	}
	err = fim.Run(config, *socketPath, debugConfig)
	if err != nil {
		log.Fatal("FATAL: ", err)
	}
}
