package container

import (
	"path/filepath"
	"reflect"
	"strings"
	"unsafe"
)

// A tiny trie implementation for paths. Splits paths by filepath.Separator and considers absolute and
// relative paths equal, e.g. "/a/b/c" is the same as "a/b/c" and is the same as "a//b/c/"
type PathTrie struct {
	// NOTE: Go has messed up semantics regarding copying values (e.g. copying slices). Making root a pointer
	// instead of a value makes the pathTrie similar to the map (the copy sees the modifications to the
	// original, unless the original was nil at the time of copying) at the very minor cost to performance.
	root *trieNode
}

type trieNode struct {
	// NOTE: An alternative to hasValue would be prohibiting nil values.
	value    interface{}
	hasValue bool

	children map[string]*trieNode
}

// Inserts value for path if none is present. The first return value is either inserted or already present
// value, the second is true if insert did happen. Path cannot be empty and cannot contain ".." (i.e. must be
// normalized).
func (t *PathTrie) Insert(path string, value interface{}) (interface{}, bool) {
	if path == "" {
		panic("empty paths are not allowed")
	}
	if t.root == nil {
		t.root = &trieNode{}
	}
	splitter := pathSplitter{path: path}
	cur := t.root
	for {
		part, ok := splitter.next()
		if !ok {
			break
		}

		next, ok := cur.children[part]
		if !ok {
			if cur.children == nil {
				cur.children = map[string]*trieNode{}
			}
			next = &trieNode{}
			// Make a copy because we do not want to keep the reference to original path.
			partCopy := copyString(part)
			cur.children[partCopy] = next
		}
		cur = next
	}
	if cur.hasValue {
		return cur.value, false
	}
	cur.value = value
	cur.hasValue = true
	return value, true
}

// Finds the closest parent to the path and returns associated value and true. If neither path or any of its
// parents are present, returns nil and false. Path cannot be empty and cannot contain ".." (i.e. must be
// normalized).
func (t *PathTrie) GetParent(path string) (interface{}, bool) {
	if path == "" {
		panic("empty paths are not allowed")
	}
	if t.root == nil {
		return nil, false
	}

	_, lastParent := findNodeFor(t.root, path)
	if lastParent == nil {
		return nil, false
	}
	return lastParent.value, true
}

// Removes the entry for path, returns true if path has been removed, false if the path was not added in the
// first place.
func (t *PathTrie) Remove(path string) bool {
	if path == "" {
		panic("empty paths are not allowed")
	}
	if t.root == nil {
		return false
	}

	node, _ := findNodeFor(t.root, path)
	if node == nil {
		return false
	}
	if node.hasValue {
		node.value = nil
		node.hasValue = false
		return true
	}
	return false
}

type TrieWalkFn func(string, interface{})

// Iterates over all elements of path trie.
func (t *PathTrie) Walk(fn TrieWalkFn) {
	if t.root == nil {
		return
	}
	// TODO: Fix for Windows. We cannot determine between UNC and regular paths there, so maybe just store the
	// full paths in each node.
	rootPath := "/"
	walkImpl(t.root, rootPath, fn)
}

// Iterates over all element of path trie starting with root (including the root).
func (t *PathTrie) WalkFrom(root string, fn TrieWalkFn) {
	if t.root == nil {
		return
	}
	node, _ := findNodeFor(t.root, root)
	if node == nil {
		return
	}
	walkImpl(node, root, fn)
}

// Iterates over all root elements of path trie (i.e. all elements without a parent).
func (t *PathTrie) WalkRoots(fn TrieWalkFn) {
	if t.root == nil {
		return
	}
	// TODO: Fix for Windows. We cannot determine between UNC and regular paths there, so maybe just store the
	// full paths in each node.
	rootPath := "/"
	walkRootsImpl(t.root, rootPath, fn)
}

// Removes every node below the roots.
func (t *PathTrie) LeaveOnlyRoots() {
	if t.root == nil {
		return
	}
	leaveOnlyRootsImpl(t.root)
}

type pathSplitter struct {
	path string
}

func (w *pathSplitter) next() (string, bool) {
	for {
		if w.path == "" {
			return "", false
		}
		nextStart := strings.IndexRune(w.path, filepath.Separator)
		if nextStart == -1 {
			part := w.path
			w.path = ""
			return part, true
		}
		part := w.path[:nextStart]
		w.path = w.path[nextStart+1:]
		if part == ".." {
			panic("paths with .. are not allowed")
		}
		// Generally empty parts would be the first (in absolute unix paths) and the last one (e.g. in
		// "/usr/"). TODO: Fix for Windows UNC paths?
		if part != "" {
			return part, true
		}
	}
}

// Returns a pair: either both elements are node for path, or the first is nil and the second is the
// bottommost parent for path (or nil).
func findNodeFor(root *trieNode, path string) (*trieNode, *trieNode) {
	splitter := pathSplitter{path: path}
	cur := root
	var lastParent *trieNode
	if cur.hasValue {
		lastParent = cur
	}
	for {
		part, ok := splitter.next()
		if !ok {
			break
		}
		cur = cur.children[part]
		if cur == nil {
			break
		}
		if cur.hasValue {
			lastParent = cur
		}
	}
	return cur, lastParent
}

func walkImpl(node *trieNode, curPath string, fn TrieWalkFn) {
	// NOTE: This method can be implemented with zero allocations, but this should be a bottleneck for this
	// particular application.
	if node.hasValue {
		fn(curPath, node.value)
	}
	for part, child := range node.children {
		walkImpl(child, filepath.Join(curPath, part), fn)
	}
}

func walkRootsImpl(node *trieNode, curPath string, fn TrieWalkFn) {
	// NOTE: This method can be implemented with zero allocations, but this should be a bottleneck for this
	// particular application.
	if node.hasValue {
		fn(curPath, node.value)
		return
	}
	for part, child := range node.children {
		walkRootsImpl(child, filepath.Join(curPath, part), fn)
	}
}

func leaveOnlyRootsImpl(node *trieNode) {
	if node.hasValue {
		node.children = nil
		return
	}
	for _, child := range node.children {
		leaveOnlyRootsImpl(child)
	}
}

func copyString(s string) string {
	// See https://groups.google.com/g/golang-nuts/c/naMCI9Jt6Qg for discussion.
	var b []byte
	h := (*reflect.SliceHeader)(unsafe.Pointer(&b))
	h.Data = (*reflect.StringHeader)(unsafe.Pointer(&s)).Data
	h.Len = len(s)
	h.Cap = len(s)
	return string(b)
}
