package fim

import (
	"sync"
	"time"
)

const doNotSleepBelow = 10 * time.Millisecond

// BandwidthLimiter allow specifying maximum bandwidth in bytes per second and then forces the clients
// to consume at most this bandwidth by putting them to sleep.
type bandwidthLimiter struct {
	bytesPerSecLimit int64
	timeNow          timeNow

	mu sync.Mutex
	// The algorithm is extremely simple: we record the time the first client arrived (startTime) and then maintain the
	// total consumed bandwidth (consumedBytes) since that time. If the consumed bandwidth / elapsed time is below the
	// limit, we reset the startTime and consumedBytes. Otherwise, we add the new request to consumedBytes and
	// compute the time after which the client may continue.
	startTime time.Time
	// The int64 is enough for 10Gb/sec bandwidth consumption to overflow the counter in 26 years. I hope we will
	// be using 128-bit computers by that time.
	consumedBytes int64
}

// A very simple interface for mocking time.Now()
type timeNow func() time.Time

func newBandwidthLimiter(bytesPerSecLimit int64) *bandwidthLimiter {
	return newBandWidthLimiterWithNow(bytesPerSecLimit, time.Now)
}

func newBandWidthLimiterWithNow(bytesPerSecLimit int64, timeNow timeNow) *bandwidthLimiter {
	return &bandwidthLimiter{
		bytesPerSecLimit: bytesPerSecLimit,
		timeNow:          timeNow,
	}
}

// WaitFor reports the amount of bandwidth the client wants to consume. If the bandwidth limit is hit, the
// method sleeps until the client is allowed to procced.
func (b *bandwidthLimiter) WaitFor(bytes int) {
	sleepFor := b.computeSleepFor(bytes)
	// Do not sleep for short amounts of time.
	if sleepFor > doNotSleepBelow {
		time.Sleep(sleepFor)
	}
}

func (b *bandwidthLimiter) computeSleepFor(bytes int) time.Duration {
	if b.bytesPerSecLimit == 0 {
		return 0
	}
	b.mu.Lock()
	defer b.mu.Unlock()

	dt := int64(float64(b.consumedBytes) * 1000.0 / float64(b.bytesPerSecLimit))
	sleepUntil := b.startTime.Add(time.Duration(dt) * time.Millisecond)
	now := b.timeNow()
	sleepFor := sleepUntil.Sub(now)

	if sleepFor <= 0 {
		b.startTime = now
		b.consumedBytes = 0
	}
	b.consumedBytes += int64(bytes)
	return sleepFor
}
