package fim

// Map category -> list of matching file paths, e.g.
// {
//   "root": [
//     "/usr/bin/*",
//     "/etc/**"
//   ]
// }
// See https://osquery.readthedocs.io/en/stable/deployment/file-integrity-monitoring/ for more examples.
//
// Symlinked files are included, but symlinked directories are ignored.
type CategoryToPaths map[string][]string

type Config struct {
	// Files to include.
	FilePaths CategoryToPaths `json:"file_paths"`
	// Files to exclude.
	ExcludePaths CategoryToPaths `json:"exclude_paths"`

	// Only compute hashes of executable files for the following categories (not supported on Windows).
	OnlyExecutables []string `json:"only_executables"`

	// Rehashing period, e.g. "5m" or "1h". Format is described in https://golang.org/pkg/time/#ParseDuration.
	//
	// Cannot be empty (we cannot rely only on file change notifier, as it is prone to missing at least some
	// of updates, e.g. writing to files via mmap and/or not working on some OSes).
	Period string `json:"period"`
	// Use file change notifier (e.g. fanotify) for hashing created/changed files ahead of schedule. Periodic
	// rehashes are run regardless. Enabled by default.
	UseNotify *bool `json:"use_notify,omitempty"`
	// Do not hash files which have been modified in the last time. For periodic rehashes this is the period
	// after ctime during which the file would not be hashed. Defaults to 11 seconds.
	DelayHashPeriod string `json:"delay_hash_period"`
	// Increase fs.inotify.max_user_watches to 100k for inotify if true. Defaults to true.
	BumpNotifyLimit *bool `json:"bump_notify_limit"`

	// Enable docker-specific functions: files in containers are matched and reported relative to the
	// container fs root (e.g. "/bin/sh" instead of "/var/lib/docker/overlay2/1234.../merged/bin/sh"),
	// container id + image id + image tags are added for those files. If not set, tries to enable
	// docker support but does not fail if docker is not present.
	EnableDocker *bool `json:"enable_docker,omitempty"`
	// Docker daemon socket. Uses default if empty.
	DockerSocket string `json:"docker_socket"`

	// Lower this process priority to run in background. If not set, defaults to true.
	LowerPriority *bool `json:"lower_priority"`
	// Limit I/O bandwidth per second, e.g. "100mb/sec". If not set, defaults to 15mb/sec
	LimitBandwidth *string `json:"limit_bandwidth"`

	// Enable extensive logging. If not set, defaults to false.
	Verbose *bool `json:"verbose"`
	// Log notification-related messages for debugging issues with rehashing after file changes. If not set,
	// defaults to false.
	VerboseNotify *bool `json:"verbose_notify"`
}
