package fim

import (
	"bufio"
	"encoding/hex"
	"fmt"
	"log"
	"os"
	"path/filepath"
)

type Debug struct {
	DumpPath        string
	Verbose         bool
	VerboseNotify   bool
	DisablePeriodic bool
}

type debugDumper struct {
	path string
}

func (d *debugDumper) writeDump(hashes *hashesState) {
	if d.path == "" {
		return
	}

	f, err := os.Create(d.path)
	if err != nil {
		log.Printf("ERROR: Could not open dump file: %v\n", err)
		return
	}
	defer f.Close()
	writer := bufio.NewWriter(f)

	for _, shard := range hashes.shards {
		func() {
			shard.mu.Lock()
			defer shard.mu.Unlock()
			for path, result := range shard.results {
				_, _ = fmt.Fprintf(writer, "%s\t%s\t%s\t(categories %v)\n", path, result.realPath,
					hex.EncodeToString(stringToByteSlice(result.sha256Value)), categoriesToSlice(result.categories))
			}
		}()
	}
	for _, shard := range hashes.shards {
		func() {
			shard.mu.Lock()
			defer shard.mu.Unlock()
			for path, result := range shard.dockerResults {
				_, _ = fmt.Fprintf(writer, "%s\t%s\t%s\t(container path %s, categories %v, container name %s, image tags %s)\n",
					path, result.realPath, hex.EncodeToString(stringToByteSlice(result.sha256Value)),
					path[result.container.containerPrefixLen:], categoriesToSlice(result.categories),
					result.container.containerNames, result.container.imageTags)
			}
		}()
	}
	_ = writer.Flush()
}

func newDebugDumper(dumpPath string) *debugDumper {
	if dumpPath == "" {
		return &debugDumper{}
	}

	dumpPath, err := filepath.Abs(dumpPath)
	if err != nil {
		log.Printf("ERROR: invalid dump path %s: %v\n", dumpPath, err)
		return &debugDumper{}
	}
	return &debugDumper{
		path: dumpPath,
	}
}
