package fim

import (
	"crypto/sha256"
	"fmt"
	"hash"
	"io"

	"golang.org/x/crypto/blake2b"

	"a.yandex-team.ru/security/osquery/extensions/osquery-fim/internal/platform"
)

type Algorithm int

const (
	Undefined Algorithm = iota
	Sha256
	Blake2b
)

const (
	// Number of fileHashers is limited by the number of CPUs, use bigger buffers.
	bufSize = 1024 * 1024
)

type fileHasher struct {
	h   hash.Hash
	buf []byte
}

func newFileHasher(algorithm Algorithm) (*fileHasher, error) {
	ret := &fileHasher{}
	var err error
	ret.h, err = newHasher(algorithm)
	if err != nil {
		return nil, err
	}
	ret.buf = make([]byte, bufSize)
	return ret, nil
}

func newHasher(algorithm Algorithm) (hash.Hash, error) {
	if algorithm == Sha256 {
		return sha256.New(), nil
	} else if algorithm == Blake2b {
		return blake2b.New256(nil)
	} else {
		return nil, fmt.Errorf("unknown algorithm %v", algorithm)
	}
}

func (fh *fileHasher) hashFile(path string, limiter *bandwidthLimiter, fileSize int64) ([]byte, int64, error) {
	fh.h.Reset()

	var written int64
	f, err := platform.OpenSequential(path)
	if err != nil {
		return nil, 0, err
	}
	defer func() {
		_ = f.Close()
	}()

	written, err = fh.readFrom(f, limiter, fileSize)
	if err != nil {
		return nil, 0, err
	}
	return fh.h.Sum(nil), written, nil
}

func (fh *fileHasher) readFrom(src io.Reader, limiter *bandwidthLimiter, fileSize int64) (int64, error) {
	written := int64(0)
	for {
		willRead := len(fh.buf)
		if fileSize < int64(willRead) {
			willRead = int(fileSize)
		}
		limiter.WaitFor(willRead)

		n, err := src.Read(fh.buf)
		if n > 0 {
			_, errW := fh.h.Write(fh.buf[0:n])
			if errW != nil {
				return written, errW
			}
			written += int64(n)
		}
		if err != nil {
			if err != io.EOF {
				return written, err
			}
			break
		}
	}
	return written, nil
}

func parseAlgorithm(s string) (Algorithm, error) {
	switch s {
	case "Sha256":
		return Sha256, nil
	case "Blake2b":
		return Blake2b, nil
	default:
		return Undefined, fmt.Errorf("unknown algorithm: %s", s)
	}
}
