package fim

import (
	"context"
	"fmt"
	"log"
	"os"
	"runtime"
	"strings"
	"time"

	"github.com/c2h5oh/datasize"
	"github.com/kolide/osquery-go"
	"github.com/kolide/osquery-go/plugin/table"

	"a.yandex-team.ru/security/osquery/extensions/osquery-fim/internal/container"
	"a.yandex-team.ru/security/osquery/extensions/osquery-fim/internal/platform"
)

const (
	defaultLimitBandwidth = 15 * datasize.MB
)

func Run(config Config, socketPath string, debugConfig Debug) error {
	platform.Init(debugConfig.Verbose)

	log.Printf("Starting with config %#v\n", config)

	if len(config.FilePaths) == 0 {
		return emptyRun(socketPath)
	}

	// TODO: Split extension in two processes (osquery-fim and osquery-fim-hasher) so that the osquery integration
	// is always responsive). Leave the tree walking in the osquery-fim-hasher. Leave the file notification
	// in osquery-fim?
	if config.LowerPriority == nil || *config.LowerPriority {
		platform.SelfLowerPriority()
	}

	if runtime.GOOS == "windows" {
		// TODO: Fix the path-related problems on Windows and drop this check (convert everything to forward
		// slashes? use custom IsAbs? what with UNC?)
		return fmt.Errorf("windows is currently not supported")
	}

	matchers, err := makePathMatchers(&config)
	if err != nil {
		return err
	}

	dockerClient, err := platform.NewDockerClient(config.EnableDocker, config.DockerSocket)
	if err != nil {
		return err
	}
	defer dockerClient.Close()

	useNotify := config.UseNotify == nil || *config.UseNotify

	period, err := time.ParseDuration(config.Period)
	if err != nil {
		return fmt.Errorf("could not parse period: %v", err)
	}

	var delayHashPeriod time.Duration
	if config.DelayHashPeriod != "" {
		delayHashPeriod, err = time.ParseDuration(config.DelayHashPeriod)
		if err != nil {
			return fmt.Errorf("could not parse period: %v", err)
		}
	} else {
		delayHashPeriod = 11 * time.Second
	}

	if config.BumpNotifyLimit == nil || *config.BumpNotifyLimit {
		err = platform.BumpNotifyLimit()
		if err != nil {
			// Continue running.
			log.Printf("ERROR: %v\n", err)
		}
	}

	limitBandwidth := defaultLimitBandwidth
	if config.LimitBandwidth != nil {
		text := strings.TrimSuffix(strings.TrimSuffix(*config.LimitBandwidth, "/sec"), "/s")
		var size datasize.ByteSize
		err = size.UnmarshalText([]byte(text))
		if err != nil {
			return fmt.Errorf("could not parse limit_bandwidth: %v", err)
		}
		limitBandwidth = size
	}
	dumper := newDebugDumper(debugConfig.DumpPath)
	runner := &runner{
		matchers:        matchers,
		onlyExecutables: container.MakeStringSet(config.OnlyExecutables),
		period:          period,
		disablePeriodic: debugConfig.DisablePeriodic,
		useNotify:       useNotify,
		verbose:         debugConfig.Verbose,
		verboseNotify:   debugConfig.VerboseNotify,
		delayHashPeriod: delayHashPeriod,
		dockerClient:    dockerClient,
		limiter:         newBandwidthLimiter(int64(limitBandwidth.Bytes())),
		dumper:          dumper,
	}
	runner.init()
	// We do not care about cleaning up the runner (open files/dirs/docker API client is automatically cleaned
	// up by the OS).
	go func() {
		err = runner.run()
		if err != nil {
			log.Fatalf("ERROR: %v", err)
		}
	}()

	// Pre-fill the tables before responding to queries.
	log.Printf("Waiting for first run to complete\n")
	runner.waitUntilFirstRun()
	log.Printf("First run completed, enabling osquery tables\n")

	var server *osquery.ExtensionManagerServer
	if socketPath != "" {
		server = initOsquery(socketPath, runner, debugConfig.Verbose)
	}

	// Copy-pasted from security/osquery/extensions/*
	idleConnsClosed := make(chan struct{})
	go func() {
		platform.WaitForSignal()

		if server != nil {
			if err := server.Shutdown(context.Background()); err != nil {
				log.Printf("ERROR: server shutdown: %v\n", err)
			}
		}

		close(idleConnsClosed)
	}()

	if server != nil {
		if err := server.Run(); err != nil {
			log.Fatalf("ERROR: server run: %v", err)
		}
	}

	<-idleConnsClosed

	log.Printf("Quitting\n")
	runner.dumper.writeDump(&runner.hashes)
	runner.logMetrics()

	return nil
}

func initOsquery(socketPath string, runner *runner, verbose bool) *osquery.ExtensionManagerServer {
	if _, err := os.Stat(socketPath); os.IsNotExist(err) {
		log.Fatalf("ERROR: can't find socket path: %s", socketPath)
	}

	server, err := osquery.NewExtensionManagerServer(extensionName, socketPath)
	if err != nil {
		log.Fatalf("ERROR: calling extension: %s\n", err)
	}

	gen := func(_ context.Context, queryContext table.QueryContext) ([]map[string]string, error) {
		startTime := time.Now()
		// NOTE: We currently ignore the queryContext, but this could greatly speed up the custom queries
		// because osquery would not have to parse megabytes of data. Maybe add support for it?
		ret := fillFileHashes(runner)
		if verbose {
			log.Printf("request: %v, result len %d in %v\n", queryContext, len(ret), time.Since(startTime))
		}
		return ret, nil
	}
	dockerGen := func(_ context.Context, queryContext table.QueryContext) ([]map[string]string, error) {
		startTime := time.Now()
		// NOTE: We currently ignore the queryContext, but this could greatly speed up the custom queries
		// because osquery would not have to parse megabytes of data. Maybe add support for it?
		ret := fillDockerFileHashes(runner)
		if verbose {
			log.Printf("request: %v, result len %d in %v\n", queryContext, len(ret), time.Since(startTime))
		}
		return ret, nil
	}
	server.RegisterPlugin(table.NewPlugin(fileHashesTableName, fileHashesTable(), gen))
	server.RegisterPlugin(table.NewPlugin(dockerFileHashesTableName, dockerFileHashesTable(), dockerGen))

	return server
}

func makePathMatchers(config *Config) ([]*pathMatcher, error) {
	// Validate ExcludePaths.
	for category := range config.ExcludePaths {
		if _, ok := config.FilePaths[category]; !ok {
			return nil, fmt.Errorf("unknown category in exclude_paths: %s", category)
		}
	}

	ret := []*pathMatcher{}
	for category, patterns := range config.FilePaths {
		excludePatterns := config.ExcludePaths[category]
		matcher, err := newPathMatcher(category, patterns, excludePatterns)
		if err != nil {
			return nil, err
		}
		ret = append(ret, matcher)
	}
	return ret, nil
}

// Run the extension, return empty tables.
func emptyRun(socketPath string) error {
	log.Printf("No files to watch, empty run\n")

	var server *osquery.ExtensionManagerServer
	if socketPath == "" {
		return nil
	}
	server, err := osquery.NewExtensionManagerServer(extensionName, socketPath)
	if err != nil {
		log.Fatalf("ERROR: calling extension: %s\n", err)
	}
	emptyGen := func(_ context.Context, _ table.QueryContext) ([]map[string]string, error) {
		return nil, nil
	}
	server.RegisterPlugin(table.NewPlugin(fileHashesTableName, fileHashesTable(), emptyGen))
	server.RegisterPlugin(table.NewPlugin(dockerFileHashesTableName, dockerFileHashesTable(), emptyGen))
	if err := server.Run(); err != nil {
		log.Fatalf("ERROR: server run: %v", err)
		return err
	}
	if err := server.Shutdown(context.Background()); err != nil {
		log.Printf("ERROR: server shutdown: %v\n", err)
		return err
	}
	return nil
}
