package fim

import (
	"fmt"
	"sync/atomic"
	"time"
)

const (
	// 4Kb
	smallFileThreshold = 4 * 1024
	// 1Mb
	mediumFileThreshold = 1024 * 1024
)

type hasherMetrics struct {
	// Small, medium and large files are defined by smallFileThreshold and mediumFileThreshold.
	numSmall  int64
	numMedium int64
	numLarge  int64

	// Processed sizes and times, size is in bytes, time in microseconds.
	totalSmallTime  int64
	totalSmallSize  int64
	totalMediumTime int64
	totalMediumSize int64
	totalLargeTime  int64
	totalLargeSize  int64
}

func (m *hasherMetrics) reset() {
	atomic.StoreInt64(&m.numSmall, 0)
	atomic.StoreInt64(&m.numMedium, 0)
	atomic.StoreInt64(&m.numLarge, 0)
	atomic.StoreInt64(&m.totalSmallTime, 0)
	atomic.StoreInt64(&m.totalSmallSize, 0)
	atomic.StoreInt64(&m.totalMediumTime, 0)
	atomic.StoreInt64(&m.totalMediumSize, 0)
	atomic.StoreInt64(&m.totalLargeTime, 0)
	atomic.StoreInt64(&m.totalLargeSize, 0)
}

func (m *hasherMetrics) add(size int64, duration time.Duration) {
	if size <= smallFileThreshold {
		atomic.AddInt64(&m.numSmall, 1)
		atomic.AddInt64(&m.totalSmallSize, size)
		atomic.AddInt64(&m.totalSmallTime, duration.Microseconds())
	} else if size <= mediumFileThreshold {
		atomic.AddInt64(&m.numMedium, 1)
		atomic.AddInt64(&m.totalMediumSize, size)
		atomic.AddInt64(&m.totalMediumTime, duration.Microseconds())
	} else {
		atomic.AddInt64(&m.numLarge, 1)
		atomic.AddInt64(&m.totalLargeSize, size)
		atomic.AddInt64(&m.totalLargeTime, duration.Microseconds())
	}
}

func (m *hasherMetrics) log() string {
	result := fmt.Sprintf("Number of small files (<=%dKb): %d, medium files (<=%dKb): %d, large files: %d\n",
		smallFileThreshold/1024, atomic.LoadInt64(&m.numSmall),
		mediumFileThreshold/1024, atomic.LoadInt64(&m.numMedium),
		atomic.LoadInt64(&m.numLarge))
	result += fmt.Sprintf("Hashing speed: %.1fMb/sec for small, %.1fMb/sec for medium, %.1fMb/sec for large files\n",
		float64(atomic.LoadInt64(&m.totalSmallSize))*1000000/(float64(atomic.LoadInt64(&m.totalSmallTime))*1024*1024),
		float64(atomic.LoadInt64(&m.totalMediumSize))*1000000/(float64(atomic.LoadInt64(&m.totalMediumTime))*1024*1024),
		float64(atomic.LoadInt64(&m.totalLargeSize))*1000000/(float64(atomic.LoadInt64(&m.totalLargeTime))*1024*1024))
	return result
}
