package fim

import (
	"encoding/hex"

	"github.com/kolide/osquery-go/plugin/table"
)

const (
	extensionName             = "osquery_fim_extension"
	fileHashesTableName       = "file_hashes"
	dockerFileHashesTableName = "docker_file_hashes"

	pathColumn          = "path"
	categoryColumn      = "category"
	sha256Column        = "sha256"
	realPathColumn      = "real_path"
	containerIDColumn   = "container_id"
	containerNameColumn = "container_name"
	imageIDColumn       = "image_id"
	imageTagsColumn     = "image_tags"
	hostPathColumn      = "host_path"
)

func fileHashesTable() []table.ColumnDefinition {
	return []table.ColumnDefinition{
		table.TextColumn(pathColumn),
		table.TextColumn(categoryColumn),
		table.TextColumn(sha256Column),
		table.TextColumn(realPathColumn),
	}
}

func dockerFileHashesTable() []table.ColumnDefinition {
	return []table.ColumnDefinition{
		table.TextColumn(pathColumn),
		table.TextColumn(categoryColumn),
		table.TextColumn(sha256Column),
		table.TextColumn(realPathColumn),
		table.TextColumn(containerIDColumn),
		table.TextColumn(containerNameColumn),
		table.TextColumn(imageIDColumn),
		table.TextColumn(imageTagsColumn),
		table.TextColumn(hostPathColumn),
	}
}

func fillFileHashes(runner *runner) []map[string]string {
	var resultSize int64
	for _, shard := range runner.hashes.shards {
		func() {
			shard.mu.Lock()
			defer shard.mu.Unlock()
			resultSize += int64(len(shard.results))
		}()
	}

	ret := make([]map[string]string, 0, resultSize)
	for _, shard := range runner.hashes.shards {
		func() {
			shard.mu.Lock()
			defer shard.mu.Unlock()

			for path, result := range shard.results {
				categories := categoriesToSlice(result.categories)
				// See comment in doHashJob.
				realPath := result.realPath
				if realPath == "" {
					realPath = path
				}
				for _, category := range categories {
					m := map[string]string{
						pathColumn:     path,
						categoryColumn: category,
						sha256Column:   hex.EncodeToString(stringToByteSlice(result.sha256Value)),
						realPathColumn: realPath,
					}
					ret = append(ret, m)
				}
			}
		}()
	}
	return ret
}

func fillDockerFileHashes(runner *runner) []map[string]string {
	var resultSize int64
	for _, shard := range runner.hashes.shards {
		func() {
			shard.mu.Lock()
			defer shard.mu.Unlock()
			resultSize += int64(len(shard.dockerResults))
		}()
	}

	ret := make([]map[string]string, 0, resultSize)
	for _, shard := range runner.hashes.shards {
		func() {
			shard.mu.Lock()
			defer shard.mu.Unlock()

			for path, result := range shard.dockerResults {
				categories := categoriesToSlice(result.categories)
				// See comment in doHashJob.
				realPath := result.realPath
				if realPath == "" {
					realPath = path
				}
				for _, category := range categories {
					m := map[string]string{
						pathColumn:          path[result.container.containerPrefixLen:],
						categoryColumn:      category,
						sha256Column:        hex.EncodeToString(stringToByteSlice(result.sha256Value)),
						realPathColumn:      realPath,
						containerIDColumn:   result.container.containerID,
						containerNameColumn: result.container.containerNames,
						imageIDColumn:       result.container.imageID,
						imageTagsColumn:     result.container.imageTags,
						hostPathColumn:      path,
					}
					ret = append(ret, m)
				}
			}
		}()
	}
	return ret
}
