package platform

import (
	"context"
	"fmt"
	"log"
	"path/filepath"
	"strings"
	"sync/atomic"
	"time"

	"github.com/docker/docker/api/types"
	"github.com/docker/docker/api/types/mount"
	"github.com/docker/docker/client"

	"a.yandex-team.ru/security/osquery/extensions/osquery-fim/internal/container"
)

// A wrapper around docker API with image and container info caching.
type DockerClient struct {
	// Is nil if docker client is not initialized.
	client *client.Client

	overlayDir string

	// Maps image id -> image repo tags.
	imagesCache atomic.Value

	// Maps directory path -> container info.
	containerDirsCache atomic.Value
}

type DockerContainerInfo struct {
	ContainerID string
	// Container names are concatenated via ';'
	ContainerNames string

	ImageID string
	// Image tags are concatenated via ';'
	ImageTags string

	MergedDir string
	AllDirs   []string

	Mounts map[string]string
}

func NewDockerClient(enableDocker *bool, dockerSocket string) (*DockerClient, error) {
	d := &DockerClient{}
	d.imagesCache.Store(map[string]string{})
	d.containerDirsCache.Store(container.PathTrie{})

	var err error
	if enableDocker == nil || *enableDocker {
		d.client, err = newDockerAPIClient(dockerSocket)
		if err != nil && enableDocker != nil {
			// Do not return an error if EnableDocker was not explicitly set to true.
			return nil, err
		}
	}
	if d.client == nil {
		log.Printf("Docker support disabled\n")
		return d, nil
	}
	log.Printf("Docker support enabled\n")

	info, err := d.client.Info(context.Background())
	if err != nil {
		return nil, err
	}
	d.overlayDir = filepath.Clean(filepath.Join(info.DockerRootDir, "overlay2"))

	return d, nil
}

func (d *DockerClient) Enabled() bool {
	return d.client != nil
}

func (d *DockerClient) GetContainerForPath(path string) (*DockerContainerInfo, bool, error) {
	// This relies on absence of symlinks, hardlinks and/or bind mounts to /var/lib/docker
	if !strings.HasPrefix(path, d.GetOverlayDir()) {
		return nil, false, nil
	}

	container, ok := d.findMatchingContainer(path)
	if ok {
		return container, true, nil
	}
	_, err := d.GetContainers()
	if err != nil {
		return nil, false, err
	}
	container, ok = d.findMatchingContainer(path)
	if ok {
		return container, true, nil
	}
	// TODO: Add a delayed queue for cases when the files for a new docker container gets written but the container is
	// not started.
	return nil, false, nil
}

// Returns containers list and fills the container/image cache.
func (d *DockerClient) GetContainers() ([]*DockerContainerInfo, error) {
	if d.client == nil {
		return nil, nil
	}

	containers, err := d.client.ContainerList(context.Background(), types.ContainerListOptions{})
	if err != nil {
		return nil, fmt.Errorf("could not list containers: %v", err)
	}

	var ret []*DockerContainerInfo
	newCache := container.PathTrie{}
	for _, c := range containers {
		info := &DockerContainerInfo{
			ContainerID:    c.ID,
			ContainerNames: strings.Join(c.Names, ";"),
			ImageID:        c.ImageID,
		}
		info.ImageTags, err = d.getImageTags(c.ImageID)
		if err != nil {
			log.Printf("ERROR: could get image: %v\n", err)
			continue
		}
		inspected, err := d.client.ContainerInspect(context.Background(), c.ID)
		if err != nil {
			log.Printf("ERROR: could not inspect container: %v\n", err)
			continue
		}
		if inspected.Driver != "overlay2" {
			log.Printf("WARNING: unknown driver %s\n", inspected.Driver)
			continue
		}
		var ok bool
		info.MergedDir, ok = inspected.GraphDriver.Data["MergedDir"]
		if !ok {
			log.Printf("WARNING: strange directory info: %v\n", inspected.GraphDriver)
			continue
		}
		// Cleaning is required by code in runner.go
		info.MergedDir = filepath.Clean(info.MergedDir)
		lowerDir, ok := inspected.GraphDriver.Data["LowerDir"]
		if !ok {
			log.Printf("WARNING: strange directory info: %v\n", inspected.GraphDriver)
			continue
		}
		upperDir, ok := inspected.GraphDriver.Data["UpperDir"]
		if !ok {
			log.Printf("WARNING: strange directory info: %v\n", inspected.GraphDriver)
			continue
		}
		info.AllDirs = append(strings.Split(lowerDir, ":"), upperDir)
		for i := range info.AllDirs {
			info.AllDirs[i] = filepath.Clean(info.AllDirs[i])
		}

		info.Mounts = map[string]string{}
		for _, m := range inspected.Mounts {
			if m.Type == mount.TypeBind {
				// TODO: Check propagation? Check consistency?
				info.Mounts[m.Source] = m.Destination
			}
			// TODO: Process mount.TypeVolume
		}

		ret = append(ret, info)
		for _, dir := range info.AllDirs {
			newCache.Insert(dir, info)
		}
	}
	d.containerDirsCache.Store(newCache)
	return ret, nil
}

func (d *DockerClient) GetOverlayDir() string {
	if d.client == nil {
		// The client should call isEnabled() before calling getOverlayDir(), use fake directory name to
		// catch mistakes.
		return "@fake@dir@"
	}
	return d.overlayDir
}

func (d *DockerClient) Close() {
	if d.client == nil {
		return
	}
	_ = d.client.Close()
}

func (d *DockerClient) getImageTags(imageID string) (string, error) {
	imagesCache := d.imagesCache.Load().(map[string]string)
	tags, ok := imagesCache[imageID]
	if ok {
		return tags, nil
	}
	imagesCache, err := d.fillImageCache()
	if err != nil {
		return "", err
	}
	tags, ok = imagesCache[imageID]
	if ok {
		return tags, nil
	}
	return "", fmt.Errorf("could not find image %s even after reloading the cache", imageID)
}

func (d *DockerClient) fillImageCache() (map[string]string, error) {
	images, err := d.client.ImageList(context.Background(), types.ImageListOptions{})
	if err != nil {
		return nil, err
	}

	newCache := make(map[string]string, len(images))
	for _, image := range images {
		newCache[image.ID] = strings.Join(image.RepoTags, ";")
	}
	d.imagesCache.Store(newCache)
	return newCache, nil
}

func (d *DockerClient) findMatchingContainer(path string) (*DockerContainerInfo, bool) {
	dirCache := d.containerDirsCache.Load().(container.PathTrie)
	v, ok := dirCache.GetParent(path)
	if v == nil {
		return nil, false
	}
	return v.(*DockerContainerInfo), ok
}

func newDockerAPIClient(dockerSocket string) (*client.Client, error) {
	var cl *client.Client
	var err error
	if dockerSocket != "" {
		cl, err = client.NewClientWithOpts(client.WithHost(dockerSocket))
	} else {
		cl, err = client.NewClientWithOpts()
	}
	if err != nil {
		return nil, err
	}
	ctx, cancelFunc := context.WithTimeout(context.Background(), time.Second)
	defer cancelFunc()
	_, err = cl.Ping(ctx)
	if err != nil {
		return nil, err
	}
	return cl, nil
}
