//go:build linux
// +build linux

package platform

import (
	"fmt"
	"log"
	"os"
	"path/filepath"
	"runtime/debug"
	"strconv"
	"strings"
	"sync"
	"unsafe"

	"golang.org/x/sys/unix"

	"a.yandex-team.ru/security/osquery/extensions/osquery-fim/internal/container"
)

// fanotify-based file watcher. Watches filesystems. The main limitations are:
//  1) fanotify cannot inspect docker containers: https://github.com/docker/for-linux/issues/309
//     > It seems like fanotify does not work across namespaces or bind mounts.
//     > https://www.spinics.net/lists/kernel/msg2110225.html
//     > But I am really not familiar with it.
//  2) fanotify does not notify about the renames/moves.

var (
	noOpenExec sync.Once
)

type fanotifier struct {
	fd int

	onModify FileNotifyFn

	addedPathsMu *sync.Mutex
	// Contains added paths.
	addedPaths container.PathTrie

	verbose bool
}

func (n *fanotifier) Add(path string) error {
	path = filepath.Clean(path)

	n.addedPathsMu.Lock()
	n.addedPaths.Insert(path, nil)
	n.addedPathsMu.Unlock()

	// Find all mountpoints below the path.
	err := n.addMountPointsForPath(path)
	if err != nil {
		return err
	}

	return nil
}

func (n *fanotifier) SetOnAttrib(callback FileNotifyFn) {
	// TODO: Fanotify supports attrib changes since Linux 5.1.
}

func (n *fanotifier) SetOnDelete(callback FileNotifyFn) {
	// TODO: Fanotify supports delete since Linux 5.1.
}

func (n *fanotifier) SetOnModify(callback FileNotifyFn) {
	n.onModify = callback
}

func (n *fanotifier) SetOnMove(callback FileMoveNotifyFn) {
	// TODO: Fanotify supports moves since Linux 5.1.
}

func (n *fanotifier) addMountPointsForPath(path string) error {
	// Find mountpoint containing the current path.
	mountPoint := getMountPoint(path)
	err := n.addMountPoint(mountPoint.path)
	if err != nil {
		return err
	}

	// Find mount points below the current path.
	var mountPointsBelow []string
	for _, mount := range getAllMountPoints() {
		if isParent(path, mount.path) && !ShouldIgnorePath(mount.path) {
			mountPointsBelow = append(mountPointsBelow, mount.path)
		}
	}

	for _, point := range mountPointsBelow {
		err := n.addMountPoint(point)
		if err != nil {
			return err
		}
	}
	return nil
}

func isParent(path string, subpath string) bool {
	if path[len(path)-1] == '/' {
		return strings.HasPrefix(subpath, path)
	} else {
		if !strings.HasPrefix(subpath, path) {
			return false
		}
		if len(path) == len(subpath) {
			return true
		}
		return subpath[len(path)] == '/'
	}
}

func (n *fanotifier) addMountPoint(mountPoint string) error {
	// We always re-add all mount points. Fanotify seems to deduplicate the watched paths. This is done
	// to avoid the inherently racy behavior of storing the added mount points locally.
	if n.verbose {
		log.Printf("Adding monitoring mount point %s\n", mountPoint)
	}
	markFlags := uint(unix.FAN_MARK_ADD | unix.FAN_MARK_MOUNT)
	// Do not monitor all modifications, this can substantially reduce the I/O performance.
	mask := uint64(unix.FAN_CLOSE_WRITE)
	// mask := uint64(unix.FAN_MODIFY | unix.FAN_CLOSE_WRITE)
	err := unix.FanotifyMark(n.fd, markFlags, mask|unix.FAN_OPEN_EXEC, 0, mountPoint)
	if err == nil {
		return nil
	}
	noOpenExec.Do(func() {
		log.Printf("WARNING: FAN_OPEN_EXEC is not supported?\n")
	})
	// We may be running an older kernel, try again without FAN_OPEN_EXEC
	return unix.FanotifyMark(n.fd, markFlags, mask, 0, mountPoint)
}

func (n *fanotifier) runEventReader() {
	buf := newByteBuffer(64 * 1024)
	for {
		// Separate function to allow recover()ing.
		n.runEventReaderIter(buf)
	}
}

func (n *fanotifier) runEventReaderIter(buf *byteBuffer) {
	defer func() {
		if r := recover(); r != nil {
			log.Printf("ERROR: panic while reading events, recovering: %v\n%s", r, string(debug.Stack()))
		}
	}()

	sizeofEvent := int(unsafe.Sizeof(unix.FanotifyEventMetadata{}))

	var err error
	if buf.remaining() == 0 {
		buf.reset()
		buf.length, err = unix.Read(n.fd, buf.buf)
		if err != nil {
			log.Printf("ERROR: error while reading from fanotify: %v\n", err)
			return
		}
	}
	if buf.remaining() < sizeofEvent {
		log.Printf("ERROR: remaining bytes in fanotify buffer: %d, required: %d\n",
			buf.remaining(), sizeofEvent)
		buf.reset()
		return
	}

	// We rely on the fact that slices are at least 8-byte aligned. From
	// https://golang.org/pkg/sync/atomic/
	//
	// > The first word in a variable or in an allocated struct, array, or slice can be relied upon to be
	// 64-bit aligned.
	event := (*unix.FanotifyEventMetadata)(unsafe.Pointer(&buf.buf[buf.offset]))
	buf.offset += sizeofEvent

	n.processEvent(event)
}

func (n *fanotifier) processEvent(event *unix.FanotifyEventMetadata) {
	if event.Vers != unix.FANOTIFY_METADATA_VERSION {
		log.Printf("ERROR: wrong metadata version from fanotify: %#v\n", event)
		return
	}
	if event.Mask&unix.FAN_Q_OVERFLOW != 0 {
		log.Printf("WARNING: overflow in fanotify\n")
		return
	}

	path, err := os.Readlink("/proc/self/fd/" + strconv.Itoa(int(event.Fd)))
	if err != nil {
		log.Printf("ERROR: could not read symlink from fanotify fd: %v\n", err)
		return
	}
	err = unix.Close(int(event.Fd))
	if err != nil {
		log.Printf("ERROR: closing fd from fanotify: %v\n", err)
	}

	// Is there a less hackish way?
	if strings.HasPrefix(path, " (deleted)") {
		return
	}

	n.addedPathsMu.Lock()
	_, ok := n.addedPaths.GetParent(path)
	n.addedPathsMu.Unlock()
	if ok {
		if n.onModify != nil {
			n.onModify(path, false)
		}
	}
}

func (n *fanotifier) subscribeToMounts() {
	// Check if the new mount is below one of the added paths.
	var addedPaths []string
	n.addedPathsMu.Lock()
	n.addedPaths.Walk(func(path string, v interface{}) {
		addedPaths = append(addedPaths, path)
	})
	n.addedPathsMu.Unlock()

	// We always add all the mount points.
	for _, path := range addedPaths {
		err := n.addMountPointsForPath(path)
		if err != nil {
			log.Printf("ERROR: Failed adding mount point for path %s: %v\n", path, err)
		}
	}
}

func NewFaNotifier(verbose bool) (FileNotifier, error) {
	fd, err := unix.FanotifyInit(unix.FAN_CLASS_NOTIF|unix.FAN_UNLIMITED_MARKS, uint(os.O_RDONLY|unix.O_LARGEFILE))
	if err != nil {
		return nil, fmt.Errorf("fanotify_init error: %v", err)
	}

	ret := &fanotifier{
		fd:           fd,
		addedPathsMu: &sync.Mutex{},
		verbose:      verbose,
	}
	ret.subscribeToMounts()

	go ret.runEventReader()
	return ret, nil
}
