package platform

import (
	"io/fs"
	"log"
	"os"
	"path/filepath"
	"runtime/debug"

	"github.com/fsnotify/fsnotify"

	"a.yandex-team.ru/security/osquery/extensions/osquery-fim/internal/container"
)

// A generic fsnotify-based watcher. Usable only for testing:
//  * uses inotify on Linux, but less efficiently than we do (watches too many event types, including
//    IN_MODIFY);
//  * uses kqueue on Darwin, which does not allow watching more than a couple thousand files (should use
//    fsevents);
//  * uses ReadDirectoryChanges instead of USN journals on Windows.
//  * does not support renames (fsnotify does not give you the previous filename)
//  * does not support removing watched top-level watched directories (i.e. the paths supplied to Add())

type fsnotifier struct {
	watcher *fsnotify.Watcher

	onAttrib FileNotifyFn
	onDelete FileNotifyFn
	onModify FileNotifyFn

	// See comment in runner().
	addPathJobCh chan fsAddPathJob
	addedPaths   container.StringSet
}

type fsAddPathJob struct {
	path  string
	errCh chan error
}

func (n *fsnotifier) Close() error {
	return n.watcher.Close()
}

func (n *fsnotifier) Add(path string) error {
	errCh := make(chan error)
	n.addPathJobCh <- fsAddPathJob{
		path:  path,
		errCh: errCh,
	}
	return <-errCh
}

func (n *fsnotifier) SetOnAttrib(callback FileNotifyFn) {
	n.onAttrib = callback
}

func (n *fsnotifier) SetOnDelete(callback FileNotifyFn) {
	n.onDelete = callback
}

func (n *fsnotifier) SetOnModify(callback FileNotifyFn) {
	n.onModify = callback
}

func (n *fsnotifier) SetOnMove(callback FileMoveNotifyFn) {
	// fsnotify does not return the original path
}

func (n *fsnotifier) run() {
	for {
		// Separate function to allow recover()ing.
		n.runIter()
	}
}

func (n *fsnotifier) runIter() {
	defer func() {
		if r := recover(); r != nil {
			log.Printf("ERROR: panic while reading events, recovering: %v\n%s", r, string(debug.Stack()))
		}
	}()

	select {
	case job := <-n.addPathJobCh:
		// The easiest way to avoid all subtle race conditions with adding/removing directories too fast
		// is to do all adds/removes from one goroutine.
		err := n.addRecursive(job.path, false)
		if job.errCh != nil {
			job.errCh <- err
		}

	case event := <-n.watcher.Events:
		switch event.Op {
		case fsnotify.Write:
			// Contrary to docs event.Name seems to always return the full path.
			if n.onModify != nil {
				n.onModify(event.Name, false)
			}
		case fsnotify.Create:
			err := n.addRecursive(event.Name, true)
			if err != nil {
				log.Printf("ERROR: failed while adding new watches: %v\n", err)
			}

			// If the file is a regular file, wait until it has been closed after writing. However, if the
			// fils is a symlink, no further events will come.
			st, err := os.Lstat(event.Name)
			if err != nil {
				log.Printf("WARNING: got error when lstat()ting newly created file in fsnotify: %s: %v",
					event.Name, err)
				return
			}
			if st.Mode()&os.ModeSymlink != 0 {
				if n.onModify != nil {
					n.onModify(event.Name, false)
				}
			}
		case fsnotify.Remove:
			n.addedPaths.Remove(event.Name)
			if n.onDelete != nil {
				n.onDelete(event.Name, isDir(event.Name))
			}
			// NOTE: Unlike e.g. inotifier, we do not support removing the top-level watched directory,
			// it will not be re-added to the watches after being deleted.
		case fsnotify.Chmod:
			if n.onAttrib != nil {
				n.onAttrib(event.Name, isDir(event.Name))
			}
		}
	case err := <-n.watcher.Errors:
		log.Printf("WARNING: got error from fsnotify: %v\n", err)
	}
}

func (n *fsnotifier) addRecursive(path string, onlyDirs bool) error {
	ok := n.addedPaths.Insert(path)
	if !ok {
		return nil
	}

	st, err := os.Stat(path)
	if err != nil {
		return err
	}
	if onlyDirs && !st.IsDir() {
		return nil
	}

	err = n.watcher.Add(path)
	if err != nil {
		return err
	}

	return filepath.WalkDir(path, func(path string, dirent fs.DirEntry, err error) error {
		if err != nil {
			log.Printf("ERROR: walking directory %s failed: %v\n", path, err)
			if dirent != nil && dirent.IsDir() {
				return filepath.SkipDir
			}
			return nil
		}

		if dirent.IsDir() {
			if ShouldIgnorePath(path) {
				return filepath.SkipDir
			}

			ok := n.addedPaths.Insert(path)
			if !ok {
				return nil
			}
			return n.watcher.Add(path)
		}
		return nil
	})
}

func isDir(path string) bool {
	st, err := os.Lstat(path)
	if err != nil {
		log.Printf("WARNING: got error when lstat()ting newly created file in fsnotify: %s: %v", path, err)
		return false
	}
	return st.IsDir()
}

func NewFsNotifier() (FileNotifier, error) {
	watcher, err := fsnotify.NewWatcher()
	if err != nil {
		return nil, err
	}
	ret := &fsnotifier{
		watcher:      watcher,
		addPathJobCh: make(chan fsAddPathJob),
	}

	go ret.run()
	return ret, nil
}
