//go:build darwin
// +build darwin

package platform

import (
	"fmt"
	"log"
	"os"
	"sync"
	"syscall"
	"time"

	"golang.org/x/sys/unix"
)

const (
	// Choose the lowest nice value on Macos.
	darwinLowPrio = 20
)

var (
	noFcntl sync.Once
)

func GetDefaultConfigPath() string {
	return ""
}

func Init(verbose bool) {
}

// Lowers current process priority.
func SelfLowerPriority() {
	err := syscall.Setpriority(syscall.PRIO_PROCESS, 0, darwinLowPrio)
	if err != nil {
		log.Printf("ERROR: Setpriority failed: %v\n", err)
		return
	}
	log.Printf("Set nice value to %d\n", darwinLowPrio)
}

func ShouldIgnorePath(path string) bool {
	return false
}

// Return file creation time.
func GetFileCreationTime(path string, fileInfo os.FileInfo) (time.Time, error) {
	if fileInfo == nil {
		var err error
		fileInfo, err = os.Stat(path)
		if err != nil {
			return time.Time{}, err
		}
	}
	st := fileInfo.Sys().(*syscall.Stat_t)
	if st == nil {
		return time.Time{}, fmt.Errorf("stat_t not present")
	}
	return timespecToTime(st.Birthtimespec), nil
}

// Darwin does not have a concept of direct I/O, but OTOH has a very nice flag which allows reading from file
// cache but not writing to file cache.
func OpenSequential(path string) (*os.File, error) {
	f, err := os.Open(path)
	if err != nil {
		return nil, err
	}
	_, err = unix.FcntlInt(f.Fd(), unix.F_NOCACHE, 1)
	if err != nil {
		noFcntl.Do(func() {
			log.Printf("warning: fcntl(F_NOCACHE) failed: %v\n", err)
		})
	}
	return f, nil
}
