//go:build linux
// +build linux

package platform

import (
	"fmt"
	"log"
	"os"
	"sync"
	"syscall"
	"time"

	"golang.org/x/sys/unix"

	"a.yandex-team.ru/security/osquery/extensions/osquery-fim/internal/container"
)

const (
	// Choose the lowest nice value on Linux.
	linuxLowPrio = 19
)

var (
	noStatxLog   sync.Once
	noFadviseLog sync.Once

	ignoreFilesystems = container.MakeStringSet([]string{
		"sysfs",
		"proc",
		"cpuset",
		"cgroup",
		"cgroup2",
		"devtmpfs",
		"configfs",
		"debugfs",
		"tracefs",
		"securityfs",
		"sockfs",
		"dax",
		"bpf",
		"pipefs",
		"hugetlbfs",
		"devpts",
		"pstore",
		"mqueue",
		"autofs",
		"binfmt_misc",
		"fusectl",
		"fuse.arc",
		"fuse.lxcfs",
		"nsfs",
	})
)

var (
	watcher mountWatcher
)

func GetDefaultConfigPath() string {
	return "/etc/osquery/osquery-fim.conf"
}

func Init(verbose bool) {
	err := watcher.fillMountPoints()
	if err != nil {
		log.Printf("ERROR: filling mount points: %v", err)
	}
	go watcher.watch(verbose)
}

// Lowers current process priority.
func SelfLowerPriority() {
	// According to setpriority(2) manpage, Linux implementation sets only the priority of calling thread,
	// so we work around that by creating a process group first.
	err := syscall.Setpgid(0, 0)
	if err != nil {
		log.Printf("ERROR: Setpgid failed: %v\n", err)
		return
	}
	err = syscall.Setpriority(syscall.PRIO_PGRP, 0, linuxLowPrio)
	if err != nil {
		log.Printf("ERROR: Setpriority failed: %v\n", err)
		return
	}
	log.Printf("Set nice value to %d\n", linuxLowPrio)
}

// Returns true if path should be ignored because it points to the pseudofile (e.g. /proc/...)
func ShouldIgnorePath(path string) bool {
	mount := getMountPoint(path)
	return ignoreFilesystems.Contains(mount.filesystem)
}

// Return file creation time.
func GetFileCreationTime(path string, fileInfo os.FileInfo) (time.Time, error) {
	// Try the newer syscall statx and resort to stat.Ctim if not available.
	var stx unix.Statx_t
	err := unix.Statx(0, path, 0, unix.STATX_BTIME, &stx)
	if err == nil {
		return statxTimestampToTime(stx.Btime), nil
	}
	if os.IsNotExist(err) {
		return time.Time{}, err
	}

	noStatxLog.Do(func() {
		log.Printf("WARNING: statx failed, reverting to stat, creation time may be incorrect: %v\n", err)
	})

	// Ctime is not exactly the best approximation for "file creation time", but this is the only one we've
	// got.
	if fileInfo == nil {
		fileInfo, err = os.Stat(path)
		if err != nil {
			return time.Time{}, err
		}
	}
	st := fileInfo.Sys().(*syscall.Stat_t)
	if st == nil {
		return time.Time{}, fmt.Errorf("stat_t not present")
	}
	return timespecToTime(st.Ctim), nil
}

func statxTimestampToTime(ts unix.StatxTimestamp) time.Time {
	return time.Unix(ts.Sec, int64(ts.Nsec))
}

// Try to setup file in "sequential read" mode, but do not enable direct I/O.
func OpenSequential(path string) (*os.File, error) {
	f, err := os.Open(path)
	if err != nil {
		return nil, err
	}
	err = unix.Fadvise(int(f.Fd()), 0, 0, unix.FADV_SEQUENTIAL)
	if err != nil {
		noFadviseLog.Do(func() {
			log.Printf("WARNING: fadvise failed: %v\n", err)
		})
	}
	return f, nil
}
