//go:build darwin || linux
// +build darwin linux

package platform

import (
	"errors"
	"os"
	"os/signal"
	"path/filepath"
	"syscall"
	"time"
)

// Common functions for both Linux and Darwin.

// Waits until SIGTERM/SIGINT.
func WaitForSignal() {
	gracefulStop := make(chan os.Signal, 1)

	signal.Notify(gracefulStop, syscall.SIGTERM)
	signal.Notify(gracefulStop, syscall.SIGINT)
	<-gracefulStop
}

func timespecToTime(ts syscall.Timespec) time.Time {
	return time.Unix(ts.Sec, ts.Nsec)
}

func IsExecutable(fileInfo os.FileInfo) bool {
	return fileInfo.Mode().Perm()&0111 != 0
}

func GetRealPath(path string) (string, error) {
	// The fastest way to both check if the file is a symlink and read the link seems to be to call readlink
	// regardless of file type (do not call lstat) and just to ignore the errors.
	realPath, err := os.Readlink(path)
	if err != nil {
		if pathErr, ok := err.(*os.PathError); ok {
			if isInvalidArgument(pathErr.Unwrap()) {
				return "", nil
			}
		}
		return "", err
	}
	if !filepath.IsAbs(realPath) {
		realPath = filepath.Join(filepath.Dir(path), realPath)
	}
	return realPath, nil
}

func isInvalidArgument(err error) bool {
	if errors.Is(err, os.ErrInvalid) {
		return true
	}
	if errno, ok := err.(syscall.Errno); ok {
		return errno == syscall.EINVAL
	}
	return false
}
