package main

import (
	"context"
	"flag"
	"fmt"
	"log"
	"os"
	"os/signal"
	"syscall"

	"github.com/kolide/osquery-go"
	"github.com/kolide/osquery-go/plugin/table"
)

const (
	AAProfilesTableName      = "apparmor_profiles"
	AAConfinementStatus      = "apparmor_confinement_status"
	EnabledLsmsTable         = "lsms"
	LsmsExtensionName        = "lsms_extension"
	YamaPtraceScopeTableName = "yama_ptrace"
	SysctlTableName          = "sysctl"
)

func main() {
	flag.Parse()
	var socketPath string
	if flag.NArg() != 1 {
		fmt.Printf("invalid path to osquery socket\n")
		os.Exit(1)
	} else {
		socketPath = flag.Arg(0)
	}

	if _, err := os.Stat(socketPath); os.IsNotExist(err) {
		fmt.Println("Can't find socket path", socketPath)
		os.Exit(1)
	}

	server, err := osquery.NewExtensionManagerServer(LsmsExtensionName, socketPath)
	if err != nil {
		log.Fatalf("Error calling extension: %s\n", err)
	}

	server.RegisterPlugin(table.NewPlugin(AAProfilesTableName, AppArmorProfilesTable(), GenerateAppArmorProfiles))
	server.RegisterPlugin(table.NewPlugin(AAConfinementStatus, AppArmorConfinedProcessesTable(), GenerateConfinedProcessesStatus))
	server.RegisterPlugin(table.NewPlugin(EnabledLsmsTable, LsmsTable(), GenerateEnabledLsms))
	server.RegisterPlugin(table.NewPlugin(YamaPtraceScopeTableName, YamaTable(), GenerateYamaPTraceScope))
	server.RegisterPlugin(table.NewPlugin(SysctlTableName, SysctlTable(), GenerateSysctlData))

	idleConnsClosed := make(chan struct{})
	go func() {
		gracefulStop := make(chan os.Signal, 1)

		signal.Notify(gracefulStop, syscall.SIGTERM)
		signal.Notify(gracefulStop, syscall.SIGINT)
		<-gracefulStop

		if err := server.Shutdown(context.Background()); err != nil {
			log.Printf("Server shutdown error: %v", err)
		}
		close(idleConnsClosed)

	}()

	if err := server.Run(); err != nil {
		log.Fatal(err)
	}
	<-idleConnsClosed
}
