package compute

import (
	"bytes"
	"encoding/json"
	"io/ioutil"
	"log"
	"net/http"
	"strings"

	"a.yandex-team.ru/security/osquery/osquery-consistency/internal/config"
	"a.yandex-team.ru/security/osquery/osquery-consistency/internal/misc"
	"a.yandex-team.ru/security/osquery/osquery-consistency/internal/vault"
)

// https://bb.yandex-team.ru/projects/CLOUD/repos/docs/browse/ru/_api-ref/compute/api-ref/Instance/list.md
// https://github.com/valyala/fastjson

type ComputeInstancesJSON struct {
	Instances []struct {
		ID          string `json:"id"`
		FolderID    string `json:"folderId"`
		CreatedAt   string `json:"createdAt"`
		Name        string `json:"name"`
		Description string `json:"description"`
		Labels      struct {
			AbcSvc string `json:"abc_svc"`
			Env    string `json:"env"`
			Layer  string `json:"layer"`
		} `json:"labels"`
		ZoneID     string `json:"zoneId"`
		PlatformID string `json:"platformId"`
		Resources  struct {
			Memory       string `json:"memory"`
			Cores        string `json:"cores"`
			CoreFraction string `json:"coreFraction"`
			Gpus         string `json:"gpus"`
		} `json:"resources"`
		Status   string `json:"status"`
		Metadata string `json:"metadata"`
		BootDisk struct {
			Mode       string `json:"mode"`
			DeviceName string `json:"deviceName"`
			AutoDelete bool   `json:"autoDelete"`
			DiskID     string `json:"diskId"`
		} `json:"bootDisk"`
		SecondaryDisks []struct {
			Mode       string `json:"mode"`
			DeviceName string `json:"deviceName"`
			AutoDelete bool   `json:"autoDelete"`
			DiskID     string `json:"diskId"`
		} `json:"secondaryDisks"`
		NetworkInterfaces []struct {
			Index            string `json:"index"`
			MacAddress       string `json:"macAddress"`
			SubnetID         string `json:"subnetId"`
			PrimaryV4Address struct {
				Address     string `json:"address"`
				OneToOneNat struct {
					Address   string `json:"address"`
					IPVersion string `json:"ipVersion"`
				} `json:"oneToOneNat"`
			} `json:"primaryV4Address"`
			PrimaryV6Address struct {
				Address     string `json:"address"`
				OneToOneNat struct {
					Address   string `json:"address"`
					IPVersion string `json:"ipVersion"`
				} `json:"oneToOneNat"`
			} `json:"primaryV6Address"`
		} `json:"networkInterfaces"`
		Fqdn             string `json:"fqdn"`
		SchedulingPolicy struct {
			Preemptible bool `json:"preemptible"`
		} `json:"schedulingPolicy"`
		ServiceAccountID string `json:"serviceAccountId"`
	} `json:"instances"`
	NextPageToken string `json:"nextPageToken"`
}

type IamTokenStruct struct {
	IamToken  string
	ExpiresAt string
}

func getIamToken(computeConfig config.ComputeConfig, computeEnviroment config.ComputeEnviroment) (iamToken IamTokenStruct) {
	computeConfig.IamHost.Hostname = strings.Replace(computeConfig.IamHost.Hostname, computeConfig.ReplaceString, computeEnviroment.URL, -1)
	client := misc.CreateHTTPClient()

	var jsonBody = []byte(`{"yandexPassportOauthToken":"` + computeConfig.OauthToken + `"}`)
	req, err := http.NewRequest("POST", computeConfig.IamHost.Hostname+computeConfig.IamHost.Path, bytes.NewBuffer(jsonBody))
	misc.ErrorCheck(err)

	resp, err := client.Do(req)
	misc.ErrorCheck(err)
	defer func() { misc.ErrorCheck(resp.Body.Close()) }()

	body, err := ioutil.ReadAll(resp.Body)
	misc.ErrorCheck(err)

	misc.QuerySentLog(computeConfig.IamHost.Hostname, resp.StatusCode)
	if resp.StatusCode != 200 {
		log.Fatal("status code not 200, check connection to IAM")
	}
	err = json.Unmarshal(body, &iamToken)
	misc.ErrorCheck(err)
	return
}

func fetchInstances(computeConfig config.ComputeConfig, computeEnviroment config.ComputeEnviroment, iamToken IamTokenStruct, currentPageToken string) (hostStatus map[string]config.ComputeHostStatus, nextPageToken string) {
	computeConfig.ComputeHost.Hostname = strings.Replace(computeConfig.ComputeHost.Hostname, computeConfig.ReplaceString, computeEnviroment.URL, -1)
	client := misc.CreateHTTPClient()

	req, err := http.NewRequest("GET", computeConfig.ComputeHost.Hostname+computeConfig.ComputeHost.Path, nil)
	misc.ErrorCheck(err)

	q := req.URL.Query()
	q.Add("folderId", computeEnviroment.FolderID)
	q.Add("pageSize", computeEnviroment.PageSize) // 500
	q.Add("pageToken", currentPageToken)
	req.URL.RawQuery = q.Encode()
	req.Header.Add("Authorization", "Bearer "+iamToken.IamToken)

	resp, err := client.Do(req)
	misc.ErrorCheck(err)
	defer func() { misc.ErrorCheck(resp.Body.Close()) }()

	body, err := ioutil.ReadAll(resp.Body)
	misc.ErrorCheck(err)

	var computeFetchedJSON ComputeInstancesJSON
	misc.QuerySentLog(computeConfig.ComputeHost.Hostname, resp.StatusCode)
	err = json.Unmarshal(body, &computeFetchedJSON)
	// log.Println(string(body))
	misc.ErrorCheck(err)

	hostStatus = make(map[string]config.ComputeHostStatus)
	for _, val := range computeFetchedJSON.Instances {
		var tmpStruct config.ComputeHostStatus
		tmpStruct.Status = val.Status
		tmpStruct.CreatedAt = val.CreatedAt
		hostStatus[val.Fqdn] = tmpStruct
	}
	nextPageToken = computeFetchedJSON.NextPageToken
	// log.Println(fetched_json["instances"].([]interface{})[0].(map[string]interface{})["fqdn"])
	return
}

func FetchInstancesDryRun() (hostStatus map[string]config.ComputeHostStatus) {
	return
}

func FetchInstances() (hostsStatus map[string]config.ComputeHostStatus) {
	log.Println("compute start")
	computeConfig := config.GetComputeConfig()
	computeConfig.OauthToken = vault.GetComputeCreds().OauthToken
	nextPageToken := ""
	hostsStatus = make(map[string]config.ComputeHostStatus)

	var fetchedStatus map[string]config.ComputeHostStatus

	for _, computeEnviroment := range computeConfig.Enviroments {
		iamToken := getIamToken(computeConfig, computeEnviroment)
		for {
			fetchedStatus, nextPageToken = fetchInstances(computeConfig, computeEnviroment, iamToken, nextPageToken)

			for k, v := range fetchedStatus {
				hostsStatus[k] = v
			}
			if nextPageToken == "" {
				break
			}
		}
	}
	log.Printf("compute done\n\n")
	return
}
