package main

import (
	"encoding/csv"
	"os"
	"sync"

	"a.yandex-team.ru/security/osquery/osquery-coverage/internal/compute"
	"a.yandex-team.ru/security/osquery/osquery-coverage/internal/conductor"
	"a.yandex-team.ru/security/osquery/osquery-coverage/internal/config"
	"a.yandex-team.ru/security/osquery/osquery-coverage/internal/configproto"
	"a.yandex-team.ru/security/osquery/osquery-coverage/internal/ichecker"
	"a.yandex-team.ru/security/osquery/osquery-coverage/internal/misc"
	"a.yandex-team.ru/security/osquery/osquery-coverage/internal/splunk"
)

func formCsv(hosts map[string]*config.HostStatus) (csvArray [][]string) {
	csvArray = [][]string{{"Hostname", "Conductor", "ComputeStatus", "ComputeCreatedAt", "SplunkLastSeen"}}
	for host, status := range hosts {
		csvArray = append(csvArray, []string{host, status.Conductor, status.ComputeStatus, status.ComputeCreatedAt, status.SplunkLastSeen})
	}
	return
}

func saveCsv(csvArray [][]string, path string) {
	misc.Logger.Info("saving csv...")
	file, err := os.Create(path)
	if err != nil {
		misc.Logger.Info("oh no, an error")
	}
	defer func() { _ = file.Close() }()

	writer := csv.NewWriter(file)
	defer writer.Flush()
	defer misc.Logger.Info(path)

	for _, value := range csvArray {
		err := writer.Write(value)
		if err != nil {
			misc.Logger.Error("error")
		}
	}
}

func doMagic(service *configproto.Service, waitgroup *sync.WaitGroup) {
	misc.Logger.Info(service.GetService())
	computeResults := compute.FetchInstances(service)
	//compute.FetchInstancesDryRun(service)
	conductorResults := conductor.FetchInstances(service)
	//conductor.FetchInstancesDryRun(service)
	splunkResults := splunk.FetchInstanses(service)
	//splunk.FetchInstansesDryRun(service)

	hosts := make(map[string]*config.HostStatus)

	for i := range conductorResults {
		if _, ok := hosts[conductorResults[i]]; ok {
			hosts[conductorResults[i]].Conductor = "true"
		} else {
			hosts[conductorResults[i]] = &config.HostStatus{
				Conductor: "true",
			}
		}
	}

	for k, v := range computeResults {
		if _, ok := hosts[k]; ok {
			hosts[k].ComputeStatus = v.Status
			hosts[k].ComputeCreatedAt = v.CreatedAt
		} else {
			hosts[k] = &config.HostStatus{
				ComputeStatus:    v.Status,
				ComputeCreatedAt: v.CreatedAt,
			}
		}
	}

	for host, status := range splunkResults {
		if _, ok := hosts[host]; ok {
			hosts[host].SplunkLastSeen = status
		} else {
			hosts[host] = &config.HostStatus{
				SplunkLastSeen: status,
			}
		}
	}

	saveCsv(formCsv(hosts), "/tmp/"+service.GetService()+".csv")
	ichecker.Check(service.GetService())
	waitgroup.Done()
}

func main() {

	services := config.GetServices()
	var waitgroup sync.WaitGroup

	for _, service := range services {
		waitgroup.Add(1)
		go doMagic(service, &waitgroup)
	}
	waitgroup.Wait()
}
