package compute

import (
	"fmt"
	"strings"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/osquery/osquery-coverage/internal/configproto"
	"a.yandex-team.ru/security/osquery/osquery-coverage/internal/misc"
)

type ComputeInstancesJSON struct {
	Instances []struct {
		ID          string `json:"id"`
		FolderID    string `json:"folderId"`
		CreatedAt   string `json:"createdAt"`
		Name        string `json:"name"`
		Description string `json:"description"`
		Labels      struct {
			AbcSvc string `json:"abc_svc"`
			Env    string `json:"env"`
			Layer  string `json:"layer"`
		} `json:"labels"`
		ZoneID     string `json:"zoneId"`
		PlatformID string `json:"platformId"`
		Resources  struct {
			Memory       string `json:"memory"`
			Cores        string `json:"cores"`
			CoreFraction string `json:"coreFraction"`
			Gpus         string `json:"gpus"`
		} `json:"resources"`
		Status   string `json:"status"`
		Metadata string `json:"metadata"`
		BootDisk struct {
			Mode       string `json:"mode"`
			DeviceName string `json:"deviceName"`
			AutoDelete bool   `json:"autoDelete"`
			DiskID     string `json:"diskId"`
		} `json:"bootDisk"`
		SecondaryDisks []struct {
			Mode       string `json:"mode"`
			DeviceName string `json:"deviceName"`
			AutoDelete bool   `json:"autoDelete"`
			DiskID     string `json:"diskId"`
		} `json:"secondaryDisks"`
		NetworkInterfaces []struct {
			Index            string `json:"index"`
			MacAddress       string `json:"macAddress"`
			SubnetID         string `json:"subnetId"`
			PrimaryV4Address struct {
				Address     string `json:"address"`
				OneToOneNat struct {
					Address   string `json:"address"`
					IPVersion string `json:"ipVersion"`
				} `json:"oneToOneNat"`
			} `json:"primaryV4Address"`
			PrimaryV6Address struct {
				Address     string `json:"address"`
				OneToOneNat struct {
					Address   string `json:"address"`
					IPVersion string `json:"ipVersion"`
				} `json:"oneToOneNat"`
			} `json:"primaryV6Address"`
		} `json:"networkInterfaces"`
		Fqdn             string `json:"fqdn"`
		SchedulingPolicy struct {
			Preemptible bool `json:"preemptible"`
		} `json:"schedulingPolicy"`
		ServiceAccountID string `json:"serviceAccountId"`
	} `json:"instances"`
	NextPageToken string `json:"nextPageToken"`
}

type IamTokenStruct struct {
	IamToken  string
	ExpiresAt string
}

func getIamToken(iamURL string, oauth string) (iamToken IamTokenStruct) {

	resp, err := misc.Client.R().
		SetBody(`{"yandexPassportOauthToken":"` + oauth + `"}`).
		SetResult(&iamToken).
		Post(iamURL)

	if err != nil {
		misc.Logger.Error("Iam backend unavailable",
			log.Error(err),
		)
	}

	if resp.StatusCode() <= 200 && resp.StatusCode() > 299 {
		misc.Logger.Error("Iam response status is not 200",
			log.String("Status:", resp.Status()),
		)
	}

	if strings.Contains(resp.String(), "fail") || strings.Contains(resp.String(), "error") {
		misc.Logger.Error("Iam token failed to obtain:",
			log.String("Response", resp.String()),
		)
	}
	return
}

func fetchInstances(computeEnviroment *configproto.ComputeEnviroment, computeURL string, iamToken IamTokenStruct, currentPageToken string) (hostStatus map[string]*configproto.ComputeHostsStatus, nextPageToken string) {
	if computeEnviroment.GetFolderid() == "baremetal" {
		misc.Logger.Info("skip, baremetal")
		return
	}

	var computeFetchedJSON ComputeInstancesJSON

	resp, err := misc.Client.R().
		SetQueryParams(map[string]string{
			"folderId":  computeEnviroment.GetFolderid(),
			"pageSize":  computeEnviroment.GetPagesize(),
			"pageToken": currentPageToken,
		}).
		//SetAuthToken(computeConfig.Compute.GetOauthtoken()).
		SetHeader("Authorization", "Bearer "+iamToken.IamToken).
		SetResult(&computeFetchedJSON).
		Get(computeURL)

	if err != nil {
		misc.Logger.Error("COMPUTE FAILED",
			log.Error(err),
		)
	}
	if resp != nil {
		fmt.Println("")
	}
	misc.Logger.Info("Request is done",
		log.String("URL", computeURL),
		log.String("Status", resp.Status()))

	if resp.StatusCode() < 200 || resp.StatusCode() > 299 {
		misc.Logger.Error(resp.String())
	}

	hostStatus = make(map[string]*configproto.ComputeHostsStatus)
	for _, val := range computeFetchedJSON.Instances {
		//		var tmpStruct config.ComputeHostStatus
		var tmpStruct configproto.ComputeHostsStatus
		tmpStruct.Status = val.Status
		tmpStruct.CreatedAt = val.CreatedAt
		hostStatus[val.Fqdn] = &tmpStruct
	}
	nextPageToken = computeFetchedJSON.NextPageToken
	return
}

func FetchInstancesDryRun(service *configproto.Service) (hostsStatus map[string]*configproto.ComputeHostsStatus) {
	return
}

func FetchInstances(service *configproto.Service) (hostsStatus map[string]*configproto.ComputeHostsStatus) {
	misc.Logger.Info("compute started")
	misc.Logger.Info("obtaining iam token")
	prodIamToken := getIamToken("https://iam.api.cloud.yandex.net/iam/v1/tokens", service.Compute.GetOauthtoken())
	preprodIamToken := getIamToken("https://iam.api.cloud-preprod.yandex.net/iam/v1/tokens", service.Compute.GetOauthtoken())
	misc.Logger.Info(prodIamToken.IamToken)
	misc.Logger.Info(preprodIamToken.IamToken)

	nextPageToken := ""
	hostsStatus = make(map[string]*configproto.ComputeHostsStatus)
	var fetchedStatus map[string]*configproto.ComputeHostsStatus

	for _, computeEnviroment := range service.Compute.GetComputeenviroment() {
		misc.Logger.Info("Compute, obtaining instances",
			log.String("folderid:", computeEnviroment.GetFolderid()),
			log.String("env", computeEnviroment.GetEnv()),
		)
		for {
			var iamToken IamTokenStruct
			var computeURL string
			if computeEnviroment.GetEnv() == "prod" {
				iamToken = prodIamToken
				computeURL = "https://compute.api.cloud.yandex.net/compute/v1/instances"
			} else if computeEnviroment.GetEnv() == "preprod" {
				iamToken = preprodIamToken
				computeURL = "https://compute.api.cloud-preprod.yandex.net/compute/v1/instances"
			}
			fetchedStatus, nextPageToken = fetchInstances(computeEnviroment, computeURL, iamToken, nextPageToken)

			for k, v := range fetchedStatus {
				hostsStatus[k] = v
			}
			if nextPageToken == "" {
				break
			}
		}
	}

	misc.Logger.Info("compute done")
	return
}
