package ichecker

import (
	"bufio"
	"encoding/csv"
	"fmt"
	"io"
	"os"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/osquery/osquery-coverage/internal/misc"
)

type HostStatus struct {
	Hostname         string
	Conductor        string
	ComputeStatus    string
	ComputeCreatedAt time.Time
	SplunkLastSeen   string
}

func checkFileCloseErr(err error) {
	if err != nil {
		misc.Logger.Error("error",
			log.Error(err),
		)
	}
}

func Check(serviceName string) {
	csvLocation := "/tmp/" + serviceName + ".csv"
	csvFile, err := os.Open(csvLocation)
	defer func() { checkFileCloseErr(csvFile.Close()) }()
	checkFileCloseErr(err)
	csvReader := csv.NewReader(bufio.NewReader(csvFile))
	var entry HostStatus

	_, err = csvReader.Read() // skip header
	checkFileCloseErr(err)
	currentTime := time.Now()

	for {
		line, err := csvReader.Read()
		if err == io.EOF {
			break
		} else if err != nil {
			checkFileCloseErr(err)
		}

		eventTime, _ := time.Parse(time.RFC3339, line[3])
		entry = HostStatus{Hostname: strings.ToLower(line[0]), Conductor: strings.ToLower(line[1]), ComputeStatus: strings.ToUpper(line[2]), ComputeCreatedAt: eventTime, SplunkLastSeen: line[4]}

		switch {
		// ================================NOALERT================================
		case entry.Conductor == "true" && entry.ComputeStatus == "RUNNING" && strings.Contains(entry.SplunkLastSeen, ":"):
		case strings.HasPrefix(entry.Hostname, "slb-adapter-"):
		case strings.Contains(entry.Hostname, "dataproc"):
		case strings.Contains(entry.Hostname, "svc.cloud-preprod.yandex.net"):
		case strings.Contains(entry.Hostname, "svc.cloud.yandex.net"):
		case strings.Contains(entry.Hostname, "on.cloud-preprod.yandex.net"):
		case entry.ComputeCreatedAt.Year() == 0001:
		case entry.Conductor == "" && entry.ComputeStatus != "" && entry.SplunkLastSeen == "":
		case entry.Conductor == "" && entry.ComputeStatus == "STOPPED" && strings.Contains(entry.SplunkLastSeen, ":"):
		case entry.Conductor != "" && entry.ComputeStatus == "STOPPED" && strings.Contains(entry.SplunkLastSeen, ":"):
		case entry.Conductor == "true" && currentTime.Sub(eventTime).Hours() < 1:
		// =================================ALERT=================================
		case entry.Conductor == "true" && entry.ComputeStatus == "RUNNING" && currentTime.Sub(eventTime).Hours() > 1:
			fmt.Println("1. ", entry.Hostname, entry.Conductor, entry.ComputeStatus, entry.ComputeCreatedAt, entry.SplunkLastSeen)
		case entry.Conductor != "true" && entry.ComputeStatus == "RUNNING" && currentTime.Sub(eventTime).Hours() > 1:
			fmt.Println("2. ", entry.Hostname, entry.Conductor, entry.ComputeStatus, entry.ComputeCreatedAt, entry.SplunkLastSeen)
		// =================================NOALERT=================================
		case entry.Conductor == "" && entry.ComputeStatus == "" && strings.Contains(entry.SplunkLastSeen, ":"):
		case entry.Conductor == "" && entry.ComputeStatus == "" && strings.Contains(entry.SplunkLastSeen, "latest"):
		case entry.Conductor == "conductor" && entry.ComputeStatus == "COMPUTE" && strings.Contains(entry.SplunkLastSeen, "SplunkLastSeen"):
		default:
			fmt.Println(entry.Hostname, entry.Conductor, entry.ComputeStatus, entry.ComputeCreatedAt, entry.SplunkLastSeen)
		}
	}
}
