package splunk

import (
	"crypto/tls"
	"encoding/csv"
	"io"
	"net/url"
	"regexp"
	"strings"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/osquery/osquery-coverage/internal/configproto"
	"a.yandex-team.ru/security/osquery/osquery-coverage/internal/misc"
)

func matchTag(tag string, responseBody []byte) (matched string) {
	tagLength := len("<" + tag + ">")
	re := regexp.MustCompile("<" + tag + ">.*</" + tag + ">")
	matched = string(re.Find(responseBody))
	matched = matched[tagLength : len(matched)-tagLength-1]
	// log.Printf("Successfully matched: %q\n", matched)
	return matched
}

func getSessionKey(httpClient *resty.Client, splunkHostname string, splunkAuthPath string, username string, password string) (sessionkey string) {
	misc.Logger.Info("trying to obtain sessionkey...")
	resp, err := httpClient.R().
		SetQueryParams(map[string]string{}).
		SetFormData(map[string]string{
			"username": username,
			"password": password,
		}).
		Post(splunkHostname + splunkAuthPath)
	if err != nil {
		misc.Logger.Info("oh no, an error!",
			log.Error(err),
		)
		//<msg type="WARN">Login failed</msg>
	}
	sessionkey = matchTag("sessionKey", resp.Body())
	misc.Logger.Info("got",
		log.String("sessionkey", sessionkey),
	)
	return sessionkey
}

func fetchSplunk(httpClient *resty.Client, service *configproto.Service, sessionkey string, splunkHostname string, splunkJobsPath string) string {

	data := url.Values{}
	data.Set("search", `|  tstats latest(_time) AS latest WHERE (index=yhids earliest=-1d data.tag=`+service.Splunkenviroment.GetQuery()+`) BY host | convert ctime(latest)`)
	data.Set("output_mode", "csv")
	//strings.NewReader(data.Encode()))

	resp, err := httpClient.R().
		//SetQueryParams(map[string]string{
		//	"search": `|  tstats latest(_time) AS latest WHERE (index=yhids earliest=-1d data.tag=`+ service.Splunkenviroment.GetQuery() + `) BY host | convert ctime(latest)`,
		//	"output_mode": "csv",
		//}).
		SetBody(data.Encode()).
		SetHeaders(map[string]string{
			"Authorization": "Splunk " + sessionkey,
			"Content-Type":  "application/x-www-form-urlencoded",
		}).
		Post(splunkHostname + splunkJobsPath)

	if err != nil || (resp.StatusCode() < 200 && resp.StatusCode() > 299) {
		misc.Logger.Error("Splunk error",
			log.Error(err),
		)
	}

	if strings.Contains(resp.String(), `<msg type="ERROR">`) ||
		(strings.Contains(resp.String(), `<msg type="FATAL">`)) {
		misc.Logger.Error(resp.String())
	}
	return resp.String()
}

func splitResult(searchResult string) (hosts map[string]string) {
	resultReader := csv.NewReader(strings.NewReader(searchResult))
	hosts = map[string]string{}

	for {
		record, err := resultReader.Read()
		if err == io.EOF {
			break
		}
		//misc.Logger.Info("?")
		hosts[record[0]] = record[1]
	}
	return
}

func FetchInstansesDryRun(service *configproto.Service) (hosts map[string]string) {
	return hosts
}

func FetchInstanses(service *configproto.Service) (hosts map[string]string) {
	misc.Logger.Info("splunk start")
	httpClient := misc.Client
	httpClient.SetTLSClientConfig(&tls.Config{InsecureSkipVerify: true})

	splunkHostname := "https://search.splunk.yandex.net:8089"
	splunkJobsPath := "/services/search/jobs/export/"
	splunkAuthPath := "/services/auth/login"
	sessionkey := getSessionKey(httpClient, splunkHostname, splunkAuthPath, service.Splunkenviroment.GetUsername(), service.Splunkenviroment.GetPassword())

	rawInstances := fetchSplunk(httpClient, service, sessionkey, splunkHostname, splunkJobsPath)

	hosts = splitResult(rawInstances)

	misc.Logger.Info("splunk done")
	return
}
