package cli

import (
	"fmt"
	"log"
	"os"
	"strconv"

	"github.com/jasonlvhit/gocron"
	"github.com/labstack/echo/v4"
	"github.com/labstack/echo/v4/middleware"
	"github.com/spf13/cobra"

	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/cache"
	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/config"
	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/handlers"
	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/workers"
)

var RunCmd = &cobra.Command{
	Use:          "run",
	SilenceUsage: true,
	Short:        "run as cli tool",
	Run:          RunAsCli,
}

var RunService = &cobra.Command{
	Use:          "run-svc",
	SilenceUsage: true,
	Short:        "run as service",
	Run:          RunAsService,
}

func init() {
	RootCmd.AddCommand(RunCmd)
	RootCmd.AddCommand(RunService)
}

func RunAsService(cmd *cobra.Command, args []string) {
	flags := cmd.Flags()
	if !flags.Lookup("config").Changed {
		fmt.Printf("Unable to read --config argument")
		os.Exit(1)
	}
	configPath, _ := flags.GetString("config")
	cfg, parseErr := config.FromFile(configPath)
	if parseErr != nil {
		fmt.Printf("Unable to read config file from %s", configPath)
		os.Exit(0)
	}
	if flags.Lookup("debug").Changed {
		cfg.GeneralConf.Debug = true
	}
	httpPort := os.Getenv("HTTP_PORT")
	var port int
	if portNum, err := strconv.Atoi(httpPort); err != nil {
		port = 80
	} else {
		port = portNum
	}

	s := echo.New()
	ccache := cache.NewCache()
	cfg.RuntimeCache = ccache
	s.Use(func(next echo.HandlerFunc) echo.HandlerFunc {
		return func(c echo.Context) error {
			cc := &config.Context{Context: c, Config: cfg}
			return next(cc)
		}
	})

	s.Use(middleware.Logger())

	s.GET("/ping", handlers.Ping)
	s.GET("/coverage/:service", handlers.Coverage)
	err := s.Start(fmt.Sprintf(":%d", port))

	if err != nil {
		log.Println(err.Error())
	}
}

func RunAsCli(cmd *cobra.Command, args []string) {
	flags := cmd.Flags()
	if !flags.Lookup("config").Changed {
		fmt.Printf("Unable to read --config argument")
		os.Exit(1)
	}
	configPath, _ := flags.GetString("config")
	cfg, parseErr := config.FromFile(configPath)
	if parseErr != nil {
		fmt.Printf("Unable to read config file from %s", configPath)
		os.Exit(0)
	}
	if flags.Lookup("debug").Changed {
		cfg.GeneralConf.Debug = true
	}
	if flags.Lookup("ignore-cron").Changed {
		//runWorkersScan(*cfg)
		workers.Run(*cfg)
		os.Exit(0)
	}
	cronInterval := cfg.GeneralConf.CronDailyPlan
	gocron.Every(uint64(cronInterval)).Days().At("10:00").Do(workers.Run, *cfg)
	_, time := gocron.NextRun()
	log.Println(time)
	<-gocron.Start()
}
