package compute

import (
	"bytes"
	"crypto/tls"
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"net/url"
	"strings"
	"time"

	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/computeproto"
	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/util"
)

const (
	Name = "compute"
)

type IamToken struct {
	IamToken  string
	ExpiresAt string
}

type InstancesJSON struct {
	Instances []struct {
		ID          string `json:"id"`
		FolderID    string `json:"folderId"`
		CreatedAt   string `json:"createdAt"`
		Name        string `json:"name"`
		Description string `json:"description"`
		Labels      struct {
			AbcSvc string `json:"abc_svc"`
			Env    string `json:"env"`
			Layer  string `json:"layer"`
		} `json:"labels"`
		ZoneID     string `json:"zoneId"`
		PlatformID string `json:"platformId"`
		Resources  struct {
			Memory       string `json:"memory"`
			Cores        string `json:"cores"`
			CoreFraction string `json:"coreFraction"`
			Gpus         string `json:"gpus"`
		} `json:"resources"`
		Status   string `json:"status"`
		Metadata string `json:"metadata"`
		BootDisk struct {
			Mode       string `json:"mode"`
			DeviceName string `json:"deviceName"`
			AutoDelete bool   `json:"autoDelete"`
			DiskID     string `json:"diskId"`
		} `json:"bootDisk"`
		SecondaryDisks []struct {
			Mode       string `json:"mode"`
			DeviceName string `json:"deviceName"`
			AutoDelete bool   `json:"autoDelete"`
			DiskID     string `json:"diskId"`
		} `json:"secondaryDisks"`
		NetworkInterfaces []struct {
			Index            string `json:"index"`
			MacAddress       string `json:"macAddress"`
			SubnetID         string `json:"subnetId"`
			PrimaryV4Address struct {
				Address     string `json:"address"`
				OneToOneNat struct {
					Address   string `json:"address"`
					IPVersion string `json:"ipVersion"`
				} `json:"oneToOneNat"`
			} `json:"primaryV4Address"`
			PrimaryV6Address struct {
				Address     string `json:"address"`
				OneToOneNat struct {
					Address   string `json:"address"`
					IPVersion string `json:"ipVersion"`
				} `json:"oneToOneNat"`
			} `json:"primaryV6Address"`
		} `json:"networkInterfaces"`
		Fqdn             string `json:"fqdn"`
		SchedulingPolicy struct {
			Preemptible bool `json:"preemptible"`
		} `json:"schedulingPolicy"`
		ServiceAccountID string `json:"serviceAccountId"`
	} `json:"instances"`
	NextPageToken string `json:"nextPageToken"`
}

type Status struct {
	Status     string
	CreatedAt  string
	Hostname   string
	InstanceID string
	//tmpStruct.Status = val.Status
	//tmpStruct.CreatedAt = val.CreatedAt
	//hostStatus[val.Fqdn] = &tmpStruct
}

type Source struct {
	OauthToken          string
	IamToken            IamToken
	ComputeJWT          string
	ComputePreprodJWT   string
	ComputeIamHost      string   `yaml:"host"`
	ComputeIamTokenPath string   `yaml:"token_path"`
	ProdFolders         []string `yaml:"prod_folders"`
	PreprodFolders      []string `yaml:"preprod_folders"`
	ComputePreprodSA    string   `yaml:"compute_preprod_sa"`
	ComputePreprodKeyID string   `yaml:"compute_preprod_key"`
	ComputeProdSA       string   `yaml:"compute_prod_sa"`
	ComputeProdKeyID    string   `yaml:"compute_prod_key"`
	ProdPrivKeyPath     string
	PreprodPrivKeyPath  string
	SplunkUser          string
	SplunkPwd           string
	SplunkAuthPath      string
	TimeDelta           int
}

const preprodEnv = "preprod"
const prodEnv = "prod"
const runningStatus = "RUNNING"

//const conductorURL = "https://c.yandex-team.ru/api/groups2hosts/"
const computeProdURL = "https://compute.api.cloud.yandex.net/compute/v1/instances"
const computePreprodURL = "https://compute.api.cloud-preprod.yandex.net/compute/v1/instances"

func GetIAMToken(computeIAMUrl string, oauthToken string) (IamToken, error) {
	log.Printf("fetching for compute\n")

	requestBody, _ := json.Marshal(map[string]interface{}{
		"yandexPassportOauthToken": oauthToken,
	})

	resp, postErr := http.Post(computeIAMUrl, "application/json", bytes.NewBuffer(requestBody))

	if postErr != nil {
		log.Printf("Unable to retrieve IAM token from %s", computeIAMUrl)
		return IamToken{}, postErr
	}

	defer func() {
		err := resp.Body.Close()
		if err != nil {
			log.Fatal(err)
		}
	}()
	iamToken := IamToken{}
	body, _ := ioutil.ReadAll(resp.Body)
	// TODO: check status code for request
	//if resp.StatusCode != 200 {
	//
	//}
	unmarshalErr := json.Unmarshal(body, &iamToken)
	return iamToken, unmarshalErr
}

func getInstances(folderID string, environment string, pageSize string, computeURL string, prodToken string, currentPageToken string) (hostStatus map[string]*computeproto.ComputeHostsStatus, nextPageToken string, computeStatus map[string]*Status) {
	var computeFetchedJSON InstancesJSON
	client := &http.Client{}
	req, err := http.NewRequest("GET", computeURL, nil)
	q := req.URL.Query()
	q.Add("folderId", folderID)
	q.Add("pageSize", pageSize)
	q.Add("pageToken", currentPageToken)
	req.URL.RawQuery = q.Encode()
	if environment == preprodEnv {
		req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", prodToken))
	} else {
		req.Header.Add("Authorization", fmt.Sprintf("Bearer %s", prodToken))
	}
	resp, respErr := client.Do(req)
	if err != nil || respErr != nil {
		log.Printf("Compute failed")
		return
	}
	if resp.StatusCode < 200 || resp.StatusCode > 299 {
		log.Printf("Compute api response code = %d for folder %s", resp.StatusCode, folderID)
		return
	}
	defer func() {
		err := resp.Body.Close()
		if err != nil {
			log.Print("Unable to read response body from compute")
		}
	}()
	body, _ := ioutil.ReadAll(resp.Body)
	unmarshalErr := json.Unmarshal(body, &computeFetchedJSON)
	if unmarshalErr != nil {
		log.Printf("Unable to unmarshal %s", string(body))
	}
	hostStatus = make(map[string]*computeproto.ComputeHostsStatus)
	computeStatus = make(map[string]*Status)

	for _, val := range computeFetchedJSON.Instances {
		var tmpStruct computeproto.ComputeHostsStatus
		var statusStruct Status
		tmpStruct.Status = val.Status
		tmpStruct.CreatedAt = val.CreatedAt
		statusStruct.CreatedAt = val.CreatedAt
		statusStruct.Status = val.Status
		statusStruct.Hostname = val.Fqdn
		statusStruct.InstanceID = val.ID
		hostStatus[val.Fqdn] = &tmpStruct
		computeStatus[val.Fqdn] = &statusStruct
	}

	nextPageToken = computeFetchedJSON.NextPageToken
	return
}

func (c *Source) getSplunkHosts(checkID bool) (map[string]bool, error) {
	out := make(map[string]bool)
	//out := make(map[string]bool)
	key, _ := util.GetSplunkSession(c.SplunkUser, c.SplunkPwd, c.SplunkAuthPath, "")
	tag := "ycloud-*"
	searchStr := fmt.Sprintf("|  tstats latest(_time) AS latest WHERE (index=yhids earliest=-%dh data.tag=\"%s\") BY host | convert ctime(latest)",
		c.TimeDelta, tag)
	if checkID {
		searchStr = fmt.Sprintf("|  tstats latest(_time) AS latest WHERE (index=yhids earliest=-%dh \"data.columns.instance_id\"!=\"\") BY data.columns.instance_id | convert ctime(latest)",
			c.TimeDelta)
	}

	log.Printf("fetching splunk\n")
	data := url.Values{}
	data.Set("search", searchStr)
	data.Set("output_mode", "csv")
	tr := &http.Transport{
		TLSClientConfig: &tls.Config{InsecureSkipVerify: true},
	}
	client := &http.Client{Transport: tr, Timeout: 480 * time.Second}

	req, _ := http.NewRequest("POST",
		"https://search.splunk.yandex.net:8089/services/search/jobs/export/", bytes.NewBufferString(data.Encode()))
	req.Header.Add("Authorization", fmt.Sprintf("Splunk %s", key))
	req.Header.Add("Content-Type", "application/x-www-form-urlencoded")
	resp, err := client.Do(req)
	log.Printf("Processing Splunk request\n")
	if err != nil {
		return nil, err
	}
	defer func() {
		err := resp.Body.Close()
		if err != nil {
			log.Fatal(err)
		}
	}()
	log.Printf("Reading Splunk response body\n")
	body, readErr := ioutil.ReadAll(resp.Body)
	if readErr != nil {
		log.Printf("Unable to read response")
		return out, readErr
	}
	sbody := string(body)
	hosts := strings.Split(sbody, "\n")
	for _, h := range hosts[1:] {
		if len(h) > 1 {
			trimmed := strings.Replace(h, "\"", "", -1)
			spl := strings.Split(trimmed, ",")
			if len(spl) > 1 && len(spl[0]) > 0 {
				out[spl[0]] = true
			} else {
				return out, errors.New("splunk: unable to parse response")
			}
		}
	}
	return out, nil
}

func GetCloudMap(cfg *Source) (map[string][]string, error) {
	return map[string][]string{
		preprodEnv: cfg.PreprodFolders,
		prodEnv:    cfg.ProdFolders,
	}, nil
}

func (c *Source) FetchHostsList() ([]string, error) {
	c.GetSourceType()
	nextPageToken := ""
	pageSize := "500"
	computeStatus := make(map[string]*computeproto.ComputeHostsStatus)
	var fetchedStatus map[string]*computeproto.ComputeHostsStatus
	var cstatus map[string]*Status
	cstatusGeneral := make(map[string]*Status)
	//hosts := make([]string, 0)
	//var iamToken string
	cloudMap, _ := GetCloudMap(c)

	log.Printf("Compute api request start")
	for env, folder := range cloudMap {
		switch env {
		case preprodEnv:
			for _, folderID := range folder {
				for {
					fetchedStatus, nextPageToken, cstatus = getInstances(folderID, preprodEnv, pageSize, computePreprodURL,
						c.ComputePreprodJWT, nextPageToken)
					for _, v := range cstatus {
						if v.Status == runningStatus {
							cstatusGeneral[v.InstanceID] = v
						}
					}
					for k, v := range fetchedStatus {
						if v.Status == runningStatus {
							computeStatus[k] = v
						}
					}
					if nextPageToken == "" {
						break
					}
				}
			}
		case prodEnv:
			for _, folderID := range folder {
				for {
					log.Printf("fetching %s\n", folderID)
					fetchedStatus, nextPageToken, cstatus = getInstances(folderID, prodEnv, pageSize, computeProdURL,
						c.ComputeJWT, nextPageToken)
					for _, v := range cstatus {
						if v.Status == runningStatus {
							cstatusGeneral[v.InstanceID] = v
						}
					}
					for k, v := range fetchedStatus {
						if v.Status == runningStatus {
							computeStatus[k] = v
						}
					}
					if nextPageToken == "" {
						break
					}
				}
			}
		}
	}
	log.Printf("Request to Splunk")
	shosts, _ := c.getSplunkHosts(true)
	shhosts, _ := c.getSplunkHosts(false)
	if len(shosts) == 0 {
		retryHosts, _ := c.getSplunkHosts(true)
		shosts = retryHosts
	}
	missed := make([]string, 0)
	for _, v := range cstatusGeneral {
		inSpkunk := shosts[v.InstanceID]
		hostsExists := shhosts[v.Hostname]
		domains := strings.Split(v.Hostname, ".")
		if len(domains) > 0 && !hostsExists {
			hostsExists = shhosts[domains[0]]
		}
		if inSpkunk {
			continue
		} else {
			if !hostsExists {
				//log.Printf("%s %s \n", v.InstanceID, v.Hostname)
				missed = append(missed, fmt.Sprintf("%s:%s", v.Hostname, v.InstanceID))
				continue
			}
		}
	}
	//hosts = missed
	return missed, nil
}

func (c *Source) FetchHosts() ([]string, error) {
	return c.FetchHostsList()
}

func (c *Source) GetSourceType() string {
	return Name
}
