package config

import (
	"fmt"
	"io/ioutil"
	"log"
	"os"

	"github.com/labstack/echo/v4"
	"gopkg.in/yaml.v2"

	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/cache"
	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/compute"
	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/conductor"
	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/resulter"
	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/service"
	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/splunk"
	"a.yandex-team.ru/security/osquery/osquery-metrics/internal/util"
)

const (
	RobotUsernameEnv      = "SQ_USER"
	RobotPasswordEnv      = "SQ_PWD"
	StTokenEnv            = "SQ_STTOKEN"
	OauthTokenEnv         = "OAUTH_TOKEN"
	HecTokenEnv           = "HEC_TOKEN"
	ComputePubKey         = "COMPUTE_PUB"
	ComputePrivKey        = "COMPUTE_PRIVATE"
	ComputePreprodPrivKey = "COMPUTE_PRIVATE_PREPROD"
	ComputeURL            = "https://iam.api.cloud.yandex.net/iam/v1/tokens"
	ComputeURLPreprod     = "https://iam.api.cloud-preprod.yandex.net/iam/v1/tokens"
)

var ConfigPath = ""

type Sources struct {
	SplunkConf    splunk.Source    `yaml:"splunk"`
	ConductorConf conductor.Source `yaml:"conductor"`
}

type (
	Config struct {
		Services      []service.ConfService `yaml:"services"`
		GeneralConf   resulter.General      `yaml:"general_conf"`
		ConductorConf conductor.Source      `yaml:"conductor_conf"`
		SplunkConf    splunk.Source         `yaml:"splunk_conf"`
		ComputeConf   compute.Source        `yaml:"compute_conf"`
		RuntimeCache  *cache.MapCache
	}
)

type Context struct {
	echo.Context
	*Config
}

func (c *Context) GetConfig() *Config {
	return c.Config
}

func FromFile(cfgPath string) (*Config, error) {
	if _, err := os.Stat(cfgPath); err != nil {
		return &Config{}, err
	}
	data, _ := ioutil.ReadFile(cfgPath)
	var conf Config
	err := yaml.Unmarshal(data, &conf)
	if err != nil {
		return &Config{}, err
	}
	robotUsername, robotPwd, oauthToken := os.Getenv(RobotUsernameEnv), os.Getenv(RobotPasswordEnv), os.Getenv(OauthTokenEnv)
	stOauthToken := os.Getenv(StTokenEnv)
	hecToken := os.Getenv(HecTokenEnv)
	//computePrivKey := os.Getenv(ComputePrivKey)
	if robotUsername == "" || robotPwd == "" {
		log.Printf("Unable to read %s or %s env\n", RobotUsernameEnv, RobotPasswordEnv)
		return &conf, fmt.Errorf("unable to read from %s or %s", RobotUsernameEnv, RobotPasswordEnv)
	}
	conf.SplunkConf.UsernameCreds = robotUsername
	conf.SplunkConf.PasswordCreds = robotPwd
	conf.GeneralConf.StToken = stOauthToken
	conf.GeneralConf.HECToken = hecToken
	conf.ComputeConf.OauthToken = oauthToken
	conf.ComputeConf.SplunkUser = robotUsername
	conf.ComputeConf.SplunkPwd = robotPwd
	conf.ComputeConf.ProdPrivKeyPath = os.Getenv(ComputePrivKey)
	conf.ComputeConf.PreprodPrivKeyPath = os.Getenv(ComputePreprodPrivKey)
	conf.ComputeConf.ComputeJWT = util.GetIAMToken(ComputeURL, conf.ComputeConf.ComputeProdSA, conf.ComputeConf.ComputeProdKeyID, conf.ComputeConf.ProdPrivKeyPath)
	conf.ComputeConf.ComputePreprodJWT = util.GetIAMToken(ComputeURLPreprod, conf.ComputeConf.ComputePreprodSA, conf.ComputeConf.ComputePreprodKeyID, conf.ComputeConf.PreprodPrivKeyPath)

	conf.ComputeConf.SplunkAuthPath = fmt.Sprintf("%s%s", conf.SplunkConf.Host, conf.SplunkConf.AuthPath)
	return &conf, nil
}
